<?php
require_once 'config.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('login.php');
}

// Get user data and premium card info
try {
    $pdo = getDBConnection();
    // *** FIX: Aliased a.account_type to financial_account_type to avoid conflict ***
    $stmt = $pdo->prepare("SELECT u.*, a.account_number, a.balance, a.account_type AS financial_account_type FROM users u LEFT JOIN accounts a ON u.id = a.user_id WHERE u.id = ? LIMIT 1");
    $stmt->execute([getCurrentUserId()]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get user's first name
    $firstName = 'User';
    if (!empty($user['full_name'])) {
        $nameParts = explode(' ', $user['full_name']);
        $firstName = $nameParts[0];
    }
    
    // Get premium card info
    $stmt = $pdo->prepare("SELECT * FROM premium_cards WHERE user_id = ?");
    $stmt->execute([getCurrentUserId()]);
    $premiumCard = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $message = '';
    $messageType = ''; // success or error
    
    // Handle card activation
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['activation_code'])) {
        $activationCode = trim($_POST['activation_code']);
        
        // Validate activation code
        if (!empty($activationCode)) {
            // *** FIX: Removed '(expiry_date IS NULL OR expiry_date > NOW())' because column does not exist ***
            $stmt = $pdo->prepare("SELECT * FROM activation_codes WHERE code = ? AND used = 0");
            $stmt->execute([$activationCode]);
            $validCode = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($validCode) {
                // Mark code as used
                $stmt = $pdo->prepare("UPDATE activation_codes SET used = 1, used_by = ?, used_at = NOW() WHERE code = ?");
                $stmt->execute([getCurrentUserId(), $activationCode]);
                
                // Generate card details
                $cardNumber = generateCardNumber($pdo);
                $cvv = sprintf("%03d", rand(0, 999));
                $expiryDate = date('Y-m-d', strtotime('+3 years'));
                
                // Create premium card record
                if ($premiumCard) {
                    $stmt = $pdo->prepare("UPDATE premium_cards SET card_number = ?, cvv = ?, expiry_date = ?, status = 'active', activated_at = NOW() WHERE user_id = ?");
                    $stmt->execute([$cardNumber, $cvv, $expiryDate, getCurrentUserId()]);
                } else {
                    $stmt = $pdo->prepare("INSERT INTO premium_cards (user_id, card_number, cvv, expiry_date, status, activated_at) VALUES (?, ?, ?, ?, 'active', NOW())");
                    $stmt->execute([getCurrentUserId(), $cardNumber, $cvv, $expiryDate]);
                }
                
                $message = 'Premium card activated successfully! Your account has been upgraded to Premium. 🎉';
                $messageType = 'success';
                
                // Refresh premium card data
                $stmt = $pdo->prepare("SELECT * FROM premium_cards WHERE user_id = ?");
                $stmt->execute([getCurrentUserId()]);
                $premiumCard = $stmt->fetch(PDO::FETCH_ASSOC);
            } else {
                $message = 'Invalid or expired activation code. Please try again or contact support. ❌';
                $messageType = 'error';
            }
        } else {
            $message = 'Please enter an activation code.';
            $messageType = 'error';
        }
    }
} catch (PDOException $e) {
    error_log("Premium Card error: " . $e->getMessage());
    $user = [];
    $premiumCard = null;
    $message = 'An error occurred. Please try again.';
    $messageType = 'error';
}

// Helper function to generate unique card number
function generateCardNumber($pdo) {
    do {
        $number = '';
        // Create 4 blocks of 4 digits
        for ($i = 0; $i < 4; $i++) {
            $number .= sprintf("%04d", rand(0, 9999));
            if ($i < 3) {
                $number .= ' '; // Add space between blocks
            }
        }
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM premium_cards WHERE card_number = ?");
        $stmt->execute([$number]);
        $exists = $stmt->fetchColumn();
    } while ($exists > 0);
    
    return $number;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Premium Card - Airwaves credit union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        /* Custom scrollbar */
        ::-webkit-scrollbar {
            width: 6px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb {
            background: #d1d5db;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb:hover {
            background: #9ca3af;
        }
        
        /* Card hover effects */
        .card-hover {
            transition: all 0.3s ease;
        }
        
        .card-hover:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        }
        
        /* Gradient cards */
        .gradient-card {
            background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 50%, #ffffff 100%);
            border: 1px solid #dcfce7;
        }
        
        /* Sidebar styling */
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        /* Premium card styling */
        .premium-card {
            background: linear-gradient(135deg, #09010f 0%, #facc15 100%);
        }
        
        /* Text shadow for better readability */
        .text-sharp {
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
        }
        
        /* Animations */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Chip styling */
        .chip {
            background: linear-gradient(135deg, #f0b913 0%, #fde68a 50%, #f0b913 100%);
            border-radius: 8px;
            height: 40px;
            width: 50px;
            border: 1px solid rgba(0,0,0,0.1);
        }
        
        .card-number {
            letter-spacing: 2px;
            font-family: 'Courier New', monospace;
        }
        
        .floating-card {
            transform-style: preserve-3d;
            perspective: 1000px;
        }
        
        .card-3d {
            transform: rotateY(5deg);
            transition: transform 0.5s ease;
        }
        
        .card-3d:hover {
            transform: rotateY(0deg);
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Hamburger Menu (Mobile) -->
                    <button id="sidebar-toggle" class="md:hidden mr-3 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                    </div>
                </div>
                
                <!-- User Info & Logout (Right Side) -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($user['full_name'] ?? 'U', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($user['full_name'] ?? 'User'); ?></span>
                    </div>
                    <a href="logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Left Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen pt-16 transition-transform -translate-x-full bg-white border-r border-gray-200 md:translate-x-0 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-home text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="wire_transfer.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-paper-plane text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Wire Transfer</span>
                    </a>
                </li>
                <li>
                    <a href="deposit.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-wallet text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Deposit</span>
                    </a>
                </li>
                <li>
                    <a href="transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-receipt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">History</span>
                    </a>
                </li>
                <li>
                    <a href="track_money.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-line text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Track Money</span>
                    </a>
                </li>
                <li>
                    <a href="add_recipient.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-user-plus text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Add Recipient</span>
                    </a>
                </li>
                 <li>
                    <a href="premiumcard.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-gem text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Premium Card</span>
                    </a>
                </li>
                <li>
                    <a href="support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer with decorative element -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="gradient-card rounded-xl p-4 text-center">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Secure Banking</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main id="main-content" class="md:ml-64 pt-16 h-full pb-24 md:pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
        
            <!-- Welcome Section - Clear and bold -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-3xl font-bold text-gray-900 text-sharp">Hi <?php echo htmlspecialchars($firstName); ?>! 👋</h1>
                <p class="text-gray-600 mt-1 font-medium">Activate your exclusive premium card with unique benefits.</p>
            </div>

            <!-- Message Alert -->
            <?php if ($message): ?>
                <div class="mb-6 p-4 rounded-lg <?php echo $messageType === 'success' ? 'bg-green-100 text-green-800 border border-green-200' : 'bg-red-100 text-red-800 border border-red-200'; ?>">
                    <div class="flex items-center">
                        <i class="fas <?php echo $messageType === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'; ?> mr-2"></i>
                        <span class="font-medium"><?php echo $message; ?></span>
                    </div>
                </div>
            <?php endif; ?>

            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6">
                <!-- Premium Card Display -->
                <div class="flex justify-center lg:justify-start">
                    <div class="floating-card w-full max-w-md">
                        <div class="premium-card text-white rounded-2xl p-6 relative overflow-hidden h-64 card-3d card-hover">
                            <!-- Card Background Pattern -->
                            <div class="absolute top-0 right-0 w-40 h-40 bg-white/20 rounded-full -mr-20 -mt-20"></div>
                            <div class="absolute bottom-0 left-0 w-32 h-32 bg-white/10 rounded-full -ml-16 -mb-16"></div>
                            
                            <div class="relative z-10 h-full flex flex-col justify-between">
                                <div class="flex justify-between items-start">
                                    <div>
                                        <div class="text-lg font-bold">AIRWAVES PREMIUM</div>
                                        <div class="text-xs opacity-80">World Elite</div>
                                    </div>
                                    <div class="chip"></div>
                                </div>
                                
                                <div class="text-xl font-semibold tracking-wider card-number">
                                    <?php if ($premiumCard && $premiumCard['status'] === 'active'): ?>
                                        <?php echo $premiumCard['card_number']; ?>
                                    <?php else: ?>
                                        **** **** **** ****
                                    <?php endif; ?>
                                </div>
                                
                                <div class="flex justify-between items-end">
                                    <div>
                                        <div class="text-xs opacity-80">CARD HOLDER</div>
                                        <div class="text-sm font-medium"><?php echo htmlspecialchars($user['full_name'] ?? 'YOUR NAME'); ?></div>
                                    </div>
                                    <div>
                                        <div class="text-xs opacity-80">EXPIRES</div>
                                        <div class="text-sm font-medium">
                                            <?php if ($premiumCard && $premiumCard['status'] === 'active'): ?>
                                                <?php echo date('m/y', strtotime($premiumCard['expiry_date'])); ?>
                                            <?php else: ?>
                                                MM/YY
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="flex items-center">
                                        <i class="fab fa-cc-visa text-4xl"></i>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Card Details (Visible only when active) -->
                        <?php if ($premiumCard && $premiumCard['status'] === 'active'): ?>
                            <div class="mt-6 gradient-card rounded-xl p-6 card-hover">
                                <h3 class="text-lg font-bold text-gray-800 mb-4 flex items-center">
                                    <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                                    Card Details
                                </h3>
                                <div class="space-y-3">
                                    <div class="flex justify-between">
                                        <span class="text-gray-600 font-medium">Card Number</span>
                                        <span class="font-mono text-gray-900"><?php echo $premiumCard['card_number']; ?></span>
                                    </div>
                                    <div class="flex justify-between">
                                        <span class="text-gray-600 font-medium">CVV</span>
                                        <span class="font-mono text-gray-900"><?php echo $premiumCard['cvv']; ?></span>
                                    </div>
                                    <div class="flex justify-between">
                                        <span class="text-gray-600 font-medium">Expiry Date</span>
                                        <span class="font-medium text-gray-900"><?php echo date('F Y', strtotime($premiumCard['expiry_date'])); ?></span>
                                    </div>
                                    <div class="flex justify-between">
                                        <span class="text-gray-600 font-medium">Status</span>
                                        <span class="font-medium text-green-600">Active</span>
                                    </div>
                                    <div class="flex justify-between">
                                        <span class="text-gray-600 font-medium">Activated On</span>
                                        <span class="font-medium text-gray-900"><?php echo date('M j, Y', strtotime($premiumCard['activated_at'])); ?></span>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Activation Section -->
                <div class="gradient-card rounded-xl p-6 card-hover animate-slide-up">
                    <h2 class="text-2xl font-bold text-gray-800 mb-2">
                        <?php if ($premiumCard && $premiumCard['status'] === 'active'): ?>
                            Your Premium Card is Active
                        <?php else: ?>
                            Activate Your Premium Card
                        <?php endif; ?>
                    </h2>
                    
                    <p class="text-gray-600 mb-6">
                        <?php if ($premiumCard && $premiumCard['status'] === 'active'): ?>
                            Congratulations! Your premium card is now active and ready to use. Enjoy exclusive benefits and rewards.
                        <?php else: ?>
                            Unlock exclusive benefits with our premium card. Enter your unique activation code provided by our team.
                        <?php endif; ?>
                    </p>
                    
                    <!-- Premium Benefits -->
                    <div class="mb-8">
                        <h3 class="text-lg font-bold text-gray-800 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Premium Benefits
                        </h3>
                        <div class="grid grid-cols-1 sm:grid-cols-2 gap-4">
                            <div class="flex items-start p-3 rounded-xl bg-green-50 border border-green-100">
                                <div class="flex-shrink-0 w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                                    <i class="fas fa-coins text-green-600"></i>
                                </div>
                                <div>
                                    <h4 class="font-semibold text-gray-900">Cashback Rewards</h4>
                                    <p class="text-sm text-gray-600">Up to 5% cashback on all purchases</p>
                                </div>
                            </div>
                            <div class="flex items-start p-3 rounded-xl bg-green-50 border border-green-100">
                                <div class="flex-shrink-0 w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                                    <i class="fas fa-plane text-green-600"></i>
                                </div>
                                <div>
                                    <h4 class="font-semibold text-gray-900">Travel Insurance</h4>
                                    <p class="text-sm text-gray-600">Comprehensive travel protection</p>
                                </div>
                            </div>
                            <div class="flex items-start p-3 rounded-xl bg-green-50 border border-green-100">
                                <div class="flex-shrink-0 w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                                    <i class="fas fa-concierge-bell text-green-600"></i>
                                </div>
                                <div>
                                    <h4 class="font-semibold text-gray-900">Priority Support</h4>
                                    <p class="text-sm text-gray-600">24/7 dedicated customer service</p>
                                </div>
                            </div>
                            <div class="flex items-start p-3 rounded-xl bg-green-50 border border-green-100">
                                <div class="flex-shrink-0 w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                                    <i class="fas fa-shield-alt text-green-600"></i>
                                </div>
                                <div>
                                    <h4 class="font-semibold text-gray-900">Enhanced Security</h4>
                                    <p class="text-sm text-gray-600">Advanced fraud protection</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Activation Form -->
                    <?php if (!$premiumCard || $premiumCard['status'] !== 'active'): ?>
                        <div class="border-t border-gray-200 pt-6">
                            <h3 class="text-lg font-bold text-gray-800 mb-4 flex items-center">
                                <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                                Enter Activation Code
                            </h3>
                            <form method="POST">
                                <div class="mb-4">
                                    <label for="activation_code" class="block text-sm font-medium text-gray-700 mb-1">Unique Activation Code</label>
                                    <input 
                                        type="text" 
                                        id="activation_code" 
                                        name="activation_code" 
                                        class="w-full px-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-transparent transition-colors" 
                                        placeholder="Enter code provided by admin"
                                        required
                                        maxlength="20"
                                    >
                                    <p class="text-xs text-gray-500 mt-1">Contact our support team if you don't have an activation code.</p>
                                </div>
                                <button 
                                    type="submit" 
                                    class="w-full py-3 px-4 bg-green-600 hover:bg-green-700 text-white font-semibold rounded-lg shadow-md transition-all focus:outline-none focus:ring-2 focus:ring-green-500 focus:ring-offset-2"
                                >
                                    Activate Premium Card
                                </button>
                            </form>
                        </div>
                    <?php else: ?>
                        <div class="border-t border-gray-200 pt-6">
                            <div class="bg-green-50 border border-green-200 rounded-lg p-4">
                                <div class="flex items-center">
                                    <i class="fas fa-check-circle text-green-500 text-xl mr-3"></i>
                                    <div>
                                        <h4 class="font-semibold text-green-800">Card Successfully Activated</h4>
                                        <p class="text-sm text-green-700">Your premium card is ready to use with all benefits unlocked.</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </main>

    <!-- Mobile Navigation -->
    <div class="fixed bottom-0 left-0 right-0 md:hidden z-50">
        <div class="bg-white rounded-t-2xl shadow-lg border-t border-gray-200">
            <div class="flex justify-around items-center py-3">
                <a href="dashboard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-home text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Dashboard</span>
                </a>
                <a href="transfer.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-exchange-alt text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Transfer</span>
                </a>
                <a href="premiumcard.php" class="flex flex-col items-center text-green-600">
                    <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mb-1">
                        <i class="fas fa-credit-card text-lg"></i>
                    </div>
                    <span class="text-xs font-semibold">Cards</span>
                </a>
                <a href="transactions.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-receipt text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">History</span>
                </a>
                <a href="profile.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                         <i class="fas fa-user-circle text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Profile</span>
                </a>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebar-toggle');
            const sidebar = document.getElementById('sidebar');

            if (sidebarToggle && sidebar) {
                sidebarToggle.addEventListener('click', function(e) {
                    e.stopPropagation(); // Prevent click from bubbling up
                    sidebar.classList.toggle('-translate-x-full');
                    sidebar.classList.toggle('translate-x-0');
                });

                // Click outside to close
                document.addEventListener('click', function(e) {
                    // Check if the sidebar is open, and the click is not on the sidebar or the toggle button
                    if (sidebar.classList.contains('translate-x-0') && !sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        sidebar.classList.add('-translate-x-full');
                        sidebar.classList.remove('translate-x-0');
                    }
                });
            }

            // Card 3D Effect
            const card = document.querySelector('.card-3d');
            if (card) {
                const isTouchDevice = () => 'ontouchstart' in window || navigator.maxTouchPoints > 0;

                if (!isTouchDevice()) {
                    const container = document.querySelector('.floating-card');
                    
                    container.addEventListener('mousemove', (e) => {
                        const rect = container.getBoundingClientRect();
                        const xAxis = (rect.width / 2 - (e.pageX - rect.left)) / 20;
                        const yAxis = ((e.pageY - rect.top) - rect.height / 2) / 20;
                        
                        card.style.transform = `rotateY(${xAxis}deg) rotateX(${yAxis}deg)`;
                    });
                    
                    container.addEventListener('mouseleave', () => {
                        card.style.transform = 'rotateY(0deg) rotateX(0deg)';
                    });
                } else {
                    card.style.transform = 'rotateY(5deg)';
                }
            }
        });
    </script>
</body>
</html>