<?php
// --- ADMIN AUTHENTICATION ---
session_start();
require_once 'config.php';

// Check if user is admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Get admin user data
$admin_id = $_SESSION['admin_id'];
$pdo = getDBConnection();

// Handle transaction actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    $transaction_id = $_POST['transaction_id'] ?? '';
    
    if ($action === 'update_status' && !empty($transaction_id)) {
        $new_status = $_POST['status'] ?? '';
        if (!empty($new_status)) {
            try {
                $stmt = $pdo->prepare("UPDATE transactions SET status = ? WHERE id = ?");
                $stmt->execute([$new_status, $transaction_id]);
                
                $_SESSION['admin_success'] = "Transaction status updated successfully!";
            } catch (Exception $e) {
                $_SESSION['admin_error'] = "Error updating transaction: " . $e->getMessage();
            }
        }
    }
    
    header('Location: admin_transactions.php');
    exit;
}

// Get all transactions with user info
try {
    $search = $_GET['search'] ?? '';
    $type_filter = $_GET['type'] ?? '';
    $status_filter = $_GET['status'] ?? '';
    $date_from = $_GET['date_from'] ?? '';
    $date_to = $_GET['date_to'] ?? '';
    
    $query = "SELECT t.*, u.full_name, u.email, u.phone 
              FROM transactions t 
              JOIN users u ON t.user_id = u.id 
              WHERE 1=1";
    
    $params = [];
    
    if (!empty($search)) {
        $query .= " AND (u.full_name LIKE ? OR u.email LIKE ? OR t.description LIKE ? OR t.recipient_account LIKE ?)";
        $search_term = "%$search%";
        $params = array_merge($params, [$search_term, $search_term, $search_term, $search_term]);
    }
    
    if (!empty($type_filter)) {
        $query .= " AND t.type = ?";
        $params[] = $type_filter;
    }
    
    if (!empty($status_filter)) {
        $query .= " AND t.status = ?";
        $params[] = $status_filter;
    }
    
    if (!empty($date_from)) {
        $query .= " AND DATE(t.created_at) >= ?";
        $params[] = $date_from;
    }
    
    if (!empty($date_to)) {
        $query .= " AND DATE(t.created_at) <= ?";
        $params[] = $date_to;
    }
    
    $query .= " ORDER BY t.created_at DESC";
    
    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get transaction statistics
    $stmt = $pdo->query("
        SELECT 
            COUNT(*) as total,
            SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
            SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
            SUM(CASE WHEN type = 'credit' THEN amount ELSE 0 END) as total_credits,
            SUM(CASE WHEN type = 'debit' THEN amount ELSE 0 END) as total_debits
        FROM transactions
    ");
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    error_log("Admin Transactions error: " . $e->getMessage());
    $transactions = [];
    $stats = ['total' => 0, 'completed' => 0, 'pending' => 0, 'failed' => 0, 'total_credits' => 0, 'total_debits' => 0];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transaction Management - Airwaves credit union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .status-completed {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        .status-pending {
            background-color: #fef3c7;
            color: #d97706;
        }
        
        .status-failed {
            background-color: #fee2e2;
            color: #dc2626;
        }
        
        .transaction-type-credit {
            border-left: 4px solid #10b981;
        }
        
        .transaction-type-debit {
            border-left: 4px solid #ef4444;
        }
        
        /* Mobile responsive styles */
        .mobile-menu-btn {
            display: none;
        }
        
        @media (max-width: 1024px) {
            .mobile-menu-btn {
                display: block;
            }
            
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 40;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0 !important;
            }
            
            .overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 30;
            }
            
            .overlay.active {
                display: block;
            }
        }
        
        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: repeat(2, 1fr) !important;
            }
            
            .filter-grid {
                grid-template-columns: 1fr !important;
                gap: 1rem !important;
            }
            
            .filter-actions {
                grid-column: 1 !important;
                flex-direction: column;
                gap: 0.75rem;
            }
            
            .table-responsive {
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }
            
            .table-responsive table {
                min-width: 900px;
            }
            
            .transaction-card {
                padding: 1rem !important;
            }
        }
        
        @media (max-width: 640px) {
            .stats-grid {
                grid-template-columns: 1fr !important;
            }
            
            .nav-container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
            
            .main-container {
                padding: 1rem !important;
            }
            
            .card-padding {
                padding: 1rem !important;
            }
            
            .filter-actions button {
                width: 100%;
            }
        }
        
        /* Animation for mobile */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Form styling improvements */
        .form-input {
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 nav-container">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Mobile menu button -->
                    <button class="mobile-menu-btn mr-4 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                        <span class="ml-2 text-sm font-medium text-gray-600 hidden sm:inline">Admin</span>
                    </div>
                </div>
                
                <!-- Admin Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($_SESSION['admin_username'] ?? 'A', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
                    </div>
                    <a href="admin_logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Mobile overlay -->
    <div class="overlay"></div>

    <!-- Left Sidebar -->
    <aside class="sidebar fixed top-0 left-0 z-40 w-64 h-screen pt-16 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="admin_dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-tachometer-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="admin_users.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-users text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">User Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_transactions.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-exchange-alt text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Transactions</span>
                    </a>
                </li>
                <li>
                    <a href="admin_support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support Tickets</span>
                    </a>
                </li>
                <li>
                    <a href="admin_funds.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-money-bill-wave text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Fund Management</span>
                    </a>
                </li>
                <!-- NEW ACTIVATION CODES LINK -->
                <li>
                    <a href="admin_activation_codes.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-key text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Activation Codes</span>
                    </a>
                </li>
                <li>
                    <a href="admin_settings.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-cogs text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">System Settings</span>
                    </a>
                </li>
                <li>
                    <a href="admin_reports.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-bar text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Reports & Analytics</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-4 text-center border border-green-100">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Admin Portal</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content ml-0 lg:ml-64 pt-16 h-full pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8 main-container">
        
            <!-- Page Header -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-2xl sm:text-3xl font-bold text-gray-900">Transaction Management 💸</h1>
                <p class="text-gray-600 mt-1 text-sm sm:text-base">Monitor and manage all financial transactions</p>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($_SESSION['admin_success'])): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-check text-green-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-green-800 text-sm sm:text-base">Success!</p>
                        <p class="text-xs sm:text-sm text-green-700"><?php echo htmlspecialchars($_SESSION['admin_success']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['admin_error'])): ?>
                <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-red-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-exclamation-triangle text-red-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-red-800 text-sm sm:text-base">Error</p>
                        <p class="text-xs sm:text-sm text-red-700"><?php echo htmlspecialchars($_SESSION['admin_error']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_error']); ?>
            <?php endif; ?>

            <!-- Transaction Statistics -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6 mb-6 sm:mb-8 stats-grid">
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-blue-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Transactions</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['total']; ?></p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-blue-100 flex items-center justify-center">
                            <i class="fas fa-exchange-alt text-blue-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-green-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Completed</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['completed']; ?></p>
                            <p class="text-xs text-green-600 font-medium mt-1">Successful</p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-green-100 flex items-center justify-center">
                            <i class="fas fa-check-circle text-green-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-yellow-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Pending</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['pending']; ?></p>
                            <p class="text-xs text-yellow-600 font-medium mt-1">Awaiting action</p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-yellow-100 flex items-center justify-center">
                            <i class="fas fa-clock text-yellow-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-red-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Failed</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $stats['failed']; ?></p>
                            <p class="text-xs text-red-600 font-medium mt-1">Requires review</p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-red-100 flex items-center justify-center">
                            <i class="fas fa-exclamation-triangle text-red-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Search and Filter -->
            <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 mb-6 animate-slide-up card-padding">
                <form method="GET" action="" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4 filter-grid">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Search</label>
                        <input type="text" name="search" value="<?php echo htmlspecialchars($search); ?>" 
                               placeholder="Name, email, description..." 
                               class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Type</label>
                        <select name="type" class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base">
                            <option value="">All Types</option>
                            <option value="credit" <?php echo $type_filter === 'credit' ? 'selected' : ''; ?>>Credit</option>
                            <option value="debit" <?php echo $type_filter === 'debit' ? 'selected' : ''; ?>>Debit</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Status</label>
                        <select name="status" class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base">
                            <option value="">All Status</option>
                            <option value="completed" <?php echo $status_filter === 'completed' ? 'selected' : ''; ?>>Completed</option>
                            <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="failed" <?php echo $status_filter === 'failed' ? 'selected' : ''; ?>>Failed</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">From Date</label>
                        <input type="date" name="date_from" value="<?php echo htmlspecialchars($date_from); ?>" 
                               class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">To Date</label>
                        <input type="date" name="date_to" value="<?php echo htmlspecialchars($date_to); ?>" 
                               class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 form-input text-sm sm:text-base">
                    </div>
                    <div class="lg:col-span-5 flex flex-col sm:flex-row justify-end space-y-3 sm:space-y-0 sm:space-x-4 filter-actions">
                        <button type="submit" class="bg-green-600 text-white px-4 sm:px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm sm:text-base shadow-sm">
                            <i class="fas fa-filter mr-2"></i>
                            Apply Filters
                        </button>
                        <a href="admin_transactions.php" class="bg-gray-200 text-gray-700 px-4 sm:px-6 py-2 rounded-lg hover:bg-gray-300 transition-colors font-medium text-sm sm:text-base text-center">
                            <i class="fas fa-times mr-2"></i>
                            Clear Filters
                        </a>
                    </div>
                </form>
            </div>

            <!-- Transactions Count -->
            <?php if (!empty($transactions)): ?>
                <div class="mb-4 text-sm text-gray-500 animate-slide-up">
                    Showing <span class="font-semibold"><?php echo count($transactions); ?></span> transaction<?php echo count($transactions) !== 1 ? 's' : ''; ?>
                    <?php if (!empty($search) || !empty($type_filter) || !empty($status_filter) || !empty($date_from) || !empty($date_to)): ?>
                        matching your filters
                    <?php endif; ?>
                </div>
            <?php endif; ?>

            <!-- Transactions Table -->
            <div class="bg-white rounded-xl shadow-sm overflow-hidden animate-slide-up">
                <?php if (!empty($transactions)): ?>
                    <div class="table-responsive">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Transaction</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Type</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Amount</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                    <th class="px-4 py-3 sm:px-6 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php foreach ($transactions as $transaction): ?>
                                    <tr class="hover:bg-gray-50 transition-colors <?php echo $transaction['type'] === 'credit' ? 'transaction-type-credit' : 'transaction-type-debit'; ?>">
                                        <td class="px-4 py-4 sm:px-6 sm:py-4">
                                            <div class="text-sm font-medium text-gray-900">#TX<?php echo str_pad($transaction['id'], 6, '0', STR_PAD_LEFT); ?></div>
                                            <div class="text-sm text-gray-500 truncate max-w-[200px]"><?php echo htmlspecialchars($transaction['description'] ?? 'No description'); ?></div>
                                            <?php if (!empty($transaction['recipient_account'])): ?>
                                                <div class="text-xs text-gray-400 truncate max-w-[200px]">To: <?php echo htmlspecialchars($transaction['recipient_account']); ?></div>
                                            <?php endif; ?>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <div class="text-sm font-medium text-gray-900 truncate max-w-[120px]"><?php echo htmlspecialchars($transaction['full_name']); ?></div>
                                            <div class="text-sm text-gray-500 truncate max-w-[120px]"><?php echo htmlspecialchars($transaction['email']); ?></div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <span class="text-sm font-medium <?php echo $transaction['type'] === 'credit' ? 'text-green-600' : 'text-red-600'; ?>">
                                                <?php echo ucfirst($transaction['type']); ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <span class="text-sm font-bold <?php echo $transaction['type'] === 'credit' ? 'text-green-600' : 'text-red-600'; ?>">
                                                <?php echo $transaction['type'] === 'credit' ? '+' : '-'; ?>$<?php echo number_format($transaction['amount'], 2); ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap">
                                            <span class="status-badge status-<?php echo $transaction['status']; ?>">
                                                <?php echo ucfirst($transaction['status']); ?>
                                            </span>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm text-gray-500">
                                            <div class="flex flex-col">
                                                <span><?php echo date('M j, Y', strtotime($transaction['created_at'])); ?></span>
                                                <span class="text-xs text-gray-400"><?php echo date('g:i A', strtotime($transaction['created_at'])); ?></span>
                                            </div>
                                        </td>
                                        <td class="px-4 py-4 sm:px-6 sm:py-4 whitespace-nowrap text-sm font-medium">
                                            <form method="POST" class="inline">
                                                <input type="hidden" name="transaction_id" value="<?php echo $transaction['id']; ?>">
                                                <input type="hidden" name="action" value="update_status">
                                                <select name="status" onchange="this.form.submit()" class="text-xs border border-gray-300 rounded px-2 py-1 focus:ring-1 focus:ring-green-500 focus:border-green-500">
                                                    <option value="completed" <?php echo $transaction['status'] === 'completed' ? 'selected' : ''; ?>>Completed</option>
                                                    <option value="pending" <?php echo $transaction['status'] === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                                    <option value="failed" <?php echo $transaction['status'] === 'failed' ? 'selected' : ''; ?>>Failed</option>
                                                </select>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-8 sm:py-12">
                        <div class="w-16 h-16 sm:w-20 sm:h-20 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                            <i class="fas fa-exchange-alt text-gray-400 text-xl sm:text-2xl"></i>
                        </div>
                        <p class="text-gray-500 font-medium text-lg sm:text-xl mb-2">No transactions found</p>
                        <p class="text-gray-400 text-sm sm:text-base max-w-md mx-auto">
                            <?php echo (!empty($search) || !empty($type_filter) || !empty($status_filter) || !empty($date_from) || !empty($date_to)) ? 
                                'Try adjusting your search filters' : 
                                'No transactions have been recorded yet'; ?>
                        </p>
                        <?php if (!empty($search) || !empty($type_filter) || !empty($status_filter) || !empty($date_from) || !empty($date_to)): ?>
                            <a href="admin_transactions.php" class="inline-block mt-4 bg-green-600 text-white px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm">
                                Clear All Filters
                            </a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Financial Summary -->
            <?php if (!empty($transactions)): ?>
                <div class="mt-6 bg-gradient-to-r from-green-50 to-blue-50 rounded-xl p-4 sm:p-6 border border-green-100 animate-slide-up">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                        <i class="fas fa-chart-bar text-green-600 mr-2"></i>
                        Financial Summary
                    </h3>
                    <div class="grid grid-cols-1 sm:grid-cols-2 gap-4 text-sm">
                        <div class="flex justify-between items-center p-3 bg-white rounded-lg border border-green-200">
                            <span class="text-gray-600 font-medium">Total Credits:</span>
                            <span class="text-green-600 font-bold">$<?php echo number_format($stats['total_credits'] ?? 0, 2); ?></span>
                        </div>
                        <div class="flex justify-between items-center p-3 bg-white rounded-lg border border-red-200">
                            <span class="text-gray-600 font-medium">Total Debits:</span>
                            <span class="text-red-600 font-bold">$<?php echo number_format($stats['total_debits'] ?? 0, 2); ?></span>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <script>
        // Mobile menu functionality
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.overlay');
            
            function toggleSidebar() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
                document.body.classList.toggle('overflow-hidden');
            }
            
            if (mobileMenuBtn) {
                mobileMenuBtn.addEventListener('click', toggleSidebar);
            }
            if (overlay) {
                overlay.addEventListener('click', toggleSidebar);
            }
            
            // Handle window resize
            function handleResize() {
                if (window.innerWidth >= 1024) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                    document.body.classList.remove('overflow-hidden');
                }
            }
            
            window.addEventListener('resize', handleResize);
            
            // Auto-hide success/error messages after 5 seconds
            setTimeout(() => {
                const messages = document.querySelectorAll('[class*="bg-green-50"], [class*="bg-red-50"]');
                messages.forEach(message => {
                    message.style.opacity = '0';
                    message.style.transition = 'opacity 0.5s ease';
                    setTimeout(() => message.remove(), 500);
                });
            }, 5000);
        });

        // Set max date for date filters to today
        document.querySelectorAll('input[type="date"]').forEach(input => {
            input.max = new Date().toISOString().split('T')[0];
        });
    </script>

</body>
</html>