<?php
require_once 'config.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('login.php');
}

// Initialize variables
$user = [];
$currentUserId = getCurrentUserId();

try {
    $pdo = getDBConnection();
    
    // Get user data and account data
    $stmt = $pdo->prepare("SELECT u.*, a.account_number, a.balance, a.account_type AS financial_account_type 
                          FROM users u 
                          LEFT JOIN accounts a ON u.id = a.user_id 
                          WHERE u.id = ? LIMIT 1");
    $stmt->execute([$currentUserId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    // If user record doesn't exist, set defaults
    if (!$user) {
         $user = ['id' => $currentUserId, 'full_name' => 'User', 'email' => '', 'account_type' => 'personal'];
    }
    
    // Check if user has premium card
    $stmt = $pdo->prepare("SELECT * FROM premium_cards WHERE user_id = ? AND status = 'active'");
    $stmt->execute([$currentUserId]);
    $premiumCard = $stmt->fetch(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Add Recipient error: " . $e->getMessage());
    // Set sane defaults to prevent HTML errors
    $user = [
        'full_name' => 'User',
        'email' => '',
        'balance' => 0,
        'account_number' => 'Error',
        'account_type' => 'personal',
        'financial_account_type' => 'Error'
    ];
    $premiumCard = null;
}

// Get user's first name
$firstName = 'User';
if (!empty($user['full_name']) && $user['full_name'] != 'User') {
    $nameParts = explode(' ', $user['full_name']);
    $firstName = $nameParts[0];
}

// Process add recipient form submission
$successMessage = '';
$errorMessage = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $accountNumber = trim($_POST['account_number'] ?? '');
    $bankName = trim($_POST['bank_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $nickname = trim($_POST['nickname'] ?? '');
    
    // Validation
    if (empty($name) || empty($accountNumber)) {
        $errorMessage = "Name and Account Number are required fields.";
    } else {
        try {
            // First, let's check if recipients table exists and create it without foreign key first
            $tableExists = $pdo->query("SHOW TABLES LIKE 'recipients'")->rowCount() > 0;
            
            if (!$tableExists) {
                // Create recipients table without foreign key first
                $pdo->exec("
                    CREATE TABLE IF NOT EXISTS recipients (
                        id INT AUTO_INCREMENT PRIMARY KEY,
                        user_id INT NOT NULL,
                        name VARCHAR(100) NOT NULL,
                        account_number VARCHAR(50) NOT NULL,
                        bank_name VARCHAR(100),
                        email VARCHAR(100),
                        phone VARCHAR(20),
                        nickname VARCHAR(50),
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        UNIQUE KEY unique_user_recipient (user_id, account_number),
                        INDEX idx_user_id (user_id)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                ");
                
                // Try to add foreign key constraint separately
                try {
                    $pdo->exec("
                        ALTER TABLE recipients 
                        ADD CONSTRAINT fk_recipients_user_id 
                        FOREIGN KEY (user_id) REFERENCES users(id) 
                        ON DELETE CASCADE
                    ");
                } catch (PDOException $fkError) {
                    error_log("Foreign key creation failed: " . $fkError->getMessage());
                    // Continue without foreign key - it's not critical for functionality
                }
            }
            
            // Check if recipient already exists for this user
            $stmt = $pdo->prepare("SELECT id FROM recipients WHERE user_id = ? AND account_number = ?");
            $stmt->execute([$currentUserId, $accountNumber]);
            $existingRecipient = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($existingRecipient) {
                $errorMessage = "A recipient with this account number already exists.";
            } else {
                // Insert new recipient
                $stmt = $pdo->prepare("
                    INSERT INTO recipients (user_id, name, account_number, bank_name, email, phone, nickname) 
                    VALUES (?, ?, ?, ?, ?, ?, ?)
                ");
                
                $stmt->execute([
                    $currentUserId,
                    $name,
                    $accountNumber,
                    $bankName,
                    $email,
                    $phone,
                    $nickname
                ]);
                
                $successMessage = "Recipient added successfully!";
                
                // Clear form fields
                $_POST = [];
            }
        } catch (PDOException $e) {
            error_log("Add recipient error: " . $e->getMessage());
            
            // More specific error messages
            if (strpos($e->getMessage(), 'foreign key constraint') !== false) {
                $errorMessage = "Database configuration error. Please contact support.";
            } else {
                $errorMessage = "An error occurred while adding the recipient. Please try again.";
            }
        }
    }
}

// Get existing recipients count
try {
    $tableExists = $pdo->query("SHOW TABLES LIKE 'recipients'")->rowCount() > 0;
    
    if ($tableExists) {
        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM recipients WHERE user_id = ?");
        $stmt->execute([$currentUserId]);
        $recipientsCount = $stmt->fetch(PDO::FETCH_ASSOC)['count'] ?? 0;
    } else {
        $recipientsCount = 0;
    }
} catch (PDOException $e) {
    $recipientsCount = 0;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Recipient - Airwaves Credit Union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        /* Custom scrollbar */
        ::-webkit-scrollbar {
            width: 6px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb {
            background: #d1d5db;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb:hover {
            background: #9ca3af;
        }
        
        /* Card hover effects */
        .card-hover {
            transition: all 0.3s ease;
        }
        
        .card-hover:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        }
        
        /* Gradient cards */
        .gradient-card {
            background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 50%, #ffffff 100%);
            border: 1px solid #dcfce7;
        }
        
        /* Sidebar styling */
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        /* Premium card styling */
        .premium-card {
            background: linear-gradient(135deg, #09010f 0%, #facc15 100%);
        }
        
        /* Text shadow for better readability */
        .text-sharp {
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
        }
        
        /* Animations */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Form styling */
        .form-input {
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Hamburger Menu (Mobile) -->
                    <button id="sidebar-toggle" class="md:hidden mr-3 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                    </div>
                </div>
                
                <!-- User Info & Logout (Right Side) -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($user['full_name'] ?? 'U', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($user['full_name'] ?? 'User'); ?></span>
                    </div>
                    <a href="logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Left Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen pt-16 transition-transform -translate-x-full bg-white border-r border-gray-200 md:translate-x-0 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-home text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="wire_transfer.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-paper-plane text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Wire Transfer</span>
                    </a>
                </li>
                <li>
                    <a href="deposit.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-wallet text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Deposit</span>
                    </a>
                </li>
                <li>
                    <a href="transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-receipt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">History</span>
                    </a>
                </li>
                <li>
                    <a href="track_money.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-line text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Track Money</span>
                    </a>
                </li>
                <li>
                    <a href="add_recipient.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-user-plus text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Add Recipient</span>
                    </a>
                </li>
                 <li>
                    <a href="premiumcard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-gem text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Premium Card</span>
                    </a>
                </li>
                <li>
                    <a href="support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer with decorative element -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="gradient-card rounded-xl p-4 text-center">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Secure Banking</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main id="main-content" class="md:ml-64 pt-16 h-full pb-24 md:pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
        
            <!-- Welcome Section - Clear and bold -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-3xl font-bold text-gray-900 text-sharp">Add Recipient 👤</h1>
                <p class="text-gray-600 mt-1 font-medium">Save recipient details for faster transfers.</p>
            </div>

            <!-- Main Content Layout -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                
                <!-- Add Recipient Form (2/3) -->
                <div class="lg:col-span-2">
                    <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            New Recipient Details
                        </h2>
                        
                        <?php if ($successMessage): ?>
                            <!-- Success Message -->
                            <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center">
                                <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center mr-4">
                                    <i class="fas fa-check text-green-600"></i>
                                </div>
                                <div>
                                    <p class="font-semibold text-green-800">Success!</p>
                                    <p class="text-sm text-green-700"><?php echo htmlspecialchars($successMessage); ?></p>
                                </div>
                            </div>
                        <?php elseif ($errorMessage): ?>
                            <!-- Error Message -->
                            <div class="mb-6 p-4 bg-red-50 border border-red-200 rounded-xl flex items-center">
                                <div class="w-10 h-10 rounded-full bg-red-100 flex items-center justify-center mr-4">
                                    <i class="fas fa-exclamation-triangle text-red-600"></i>
                                </div>
                                <div>
                                    <p class="font-semibold text-red-800">Error</p>
                                    <p class="text-sm text-red-700"><?php echo htmlspecialchars($errorMessage); ?></p>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" action="add_recipient.php">
                            <div class="space-y-5">
                                <!-- Recipient Name -->
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">
                                        Full Name <span class="text-red-500">*</span>
                                    </label>
                                    <input type="text" name="name" value="<?php echo htmlspecialchars($_POST['name'] ?? ''); ?>" 
                                           placeholder="Enter recipient's full name" 
                                           class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" required>
                                </div>
                                
                                <!-- Account Number -->
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">
                                        Account Number <span class="text-red-500">*</span>
                                    </label>
                                    <input type="text" name="account_number" value="<?php echo htmlspecialchars($_POST['account_number'] ?? ''); ?>" 
                                           placeholder="Enter account number" 
                                           class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" required>
                                </div>
                                
                                <!-- Bank Name -->
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Bank Name</label>
                                    <input type="text" name="bank_name" value="<?php echo htmlspecialchars($_POST['bank_name'] ?? ''); ?>" 
                                           placeholder="Enter bank name" 
                                           class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none">
                                </div>
                                
                                <!-- Email and Phone -->
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Email Address</label>
                                        <input type="email" name="email" value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>" 
                                               placeholder="Enter email address" 
                                               class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none">
                                    </div>
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Phone Number</label>
                                        <input type="tel" name="phone" value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>" 
                                               placeholder="Enter phone number" 
                                               class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none">
                                    </div>
                                </div>
                                
                                <!-- Nickname -->
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Nickname (Optional)</label>
                                    <input type="text" name="nickname" value="<?php echo htmlspecialchars($_POST['nickname'] ?? ''); ?>" 
                                           placeholder="e.g., Mom, Business Partner, etc." 
                                           class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none">
                                    <p class="text-xs text-gray-500 mt-2">This will help you identify the recipient quickly.</p>
                                </div>
                                
                                <!-- Submit Button -->
                                <div class="pt-4">
                                    <button type="submit" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-3 px-4 rounded-xl transition-colors duration-300 flex items-center justify-center">
                                        <i class="fas fa-user-plus mr-2"></i>
                                        Add Recipient
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Sidebar Column (1/3) -->
                <div class="lg:col-span-1 space-y-6">
                    
                    <!-- Recipients Summary -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm border border-gray-100 animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Your Recipients
                        </h2>
                        
                        <div class="text-center py-4">
                            <div class="w-16 h-16 mx-auto mb-4 bg-green-100 rounded-full flex items-center justify-center">
                                <i class="fas fa-users text-green-600 text-2xl"></i>
                            </div>
                            <p class="text-3xl font-bold text-gray-900"><?php echo $recipientsCount; ?></p>
                            <p class="text-gray-600 font-medium">Saved Recipients</p>
                        </div>
                        
                        <div class="mt-4">
                            <a href="wire_transfer.php" class="w-full flex justify-center items-center py-3 px-4 border border-gray-300 rounded-xl text-sm font-semibold text-gray-700 bg-white hover:bg-gray-50 hover:border-green-500 transition-all duration-300 group">
                                <i class="fas fa-paper-plane mr-2"></i>
                                Make a Transfer
                                <i class="fas fa-arrow-right ml-2 group-hover:translate-x-1 transition-transform"></i>
                            </a>
                        </div>
                    </div>
                    
                    <!-- Benefits of Saving Recipients -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm border border-gray-100 animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Benefits
                        </h2>
                        
                        <div class="space-y-4">
                            <div class="flex items-start">
                                <div class="w-8 h-8 rounded-lg bg-green-100 flex items-center justify-center mr-3 mt-1 flex-shrink-0">
                                    <i class="fas fa-bolt text-green-600"></i>
                                </div>
                                <div>
                                    <p class="text-sm font-semibold text-gray-900">Faster Transfers</p>
                                    <p class="text-xs text-gray-500">No need to re-enter details every time</p>
                                </div>
                            </div>
                            
                            <div class="flex items-start">
                                <div class="w-8 h-8 rounded-lg bg-green-100 flex items-center justify-center mr-3 mt-1 flex-shrink-0">
                                    <i class="fas fa-check-circle text-green-600"></i>
                                </div>
                                <div>
                                    <p class="text-sm font-semibold text-gray-900">Fewer Errors</p>
                                    <p class="text-xs text-gray-500">Eliminate typos in account numbers</p>
                                </div>
                            </div>
                            
                            <div class="flex items-start">
                                <div class="w-8 h-8 rounded-lg bg-green-100 flex items-center justify-center mr-3 mt-1 flex-shrink-0">
                                    <i class="fas fa-history text-green-600"></i>
                                </div>
                                <div>
                                    <p class="text-sm font-semibold text-gray-900">Transaction History</p>
                                    <p class="text-xs text-gray-500">Track transfers to each recipient</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Mobile Navigation -->
    <div class="fixed bottom-0 left-0 right-0 md:hidden z-50">
        <div class="bg-white rounded-t-2xl shadow-lg border-t border-gray-200">
            <div class="flex justify-around items-center py-3">
                <a href="dashboard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-home text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Dashboard</span>
                </a>
                <a href="wire_transfer.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-paper-plane text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Transfer</span>
                </a>
                <a href="premiumcard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-credit-card text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Cards</span>
                </a>
                <a href="transactions.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-receipt text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">History</span>
                </a>
                <a href="add_recipient.php" class="flex flex-col items-center text-green-600">
                    <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mb-1">
                         <i class="fas fa-user-plus text-lg"></i>
                    </div>
                    <span class="text-xs font-semibold">Recipients</span>
                </a>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebar-toggle');
            const sidebar = document.getElementById('sidebar');

            if (sidebarToggle && sidebar) {
                sidebarToggle.addEventListener('click', function(e) {
                    e.stopPropagation(); // Prevent click from bubbling up
                    sidebar.classList.toggle('-translate-x-full');
                    sidebar.classList.toggle('translate-x-0');
                });

                // Click outside to close
                document.addEventListener('click', function(e) {
                    // Check if the sidebar is open, and the click is not on the sidebar or the toggle button
                    if (sidebar.classList.contains('translate-x-0') && !sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        sidebar.classList.add('-translate-x-full');
                        sidebar.classList.remove('translate-x-0');
                    }
                });
            }
        });
    </script>

</body>
</html>