<?php
// --- SESSION START ---
session_start();
require_once 'config.php';

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('login.php');
}

// Initialize variables
$user = [];
$currentUserId = getCurrentUserId();

try {
    $pdo = getDBConnection();
    
    // Get user data and account data
    $stmt = $pdo->prepare("SELECT u.*, a.account_number, a.balance, a.account_type AS financial_account_type 
                          FROM users u 
                          LEFT JOIN accounts a ON u.id = a.user_id 
                          WHERE u.id = ? LIMIT 1");
    $stmt->execute([$currentUserId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    // If user record doesn't exist, set defaults
    if (!$user) {
         $user = ['id' => $currentUserId, 'full_name' => 'User', 'email' => 'your-email@example.com', 'account_type' => 'personal'];
    }
    
    // Check if user has premium card
    $stmt = $pdo->prepare("SELECT * FROM premium_cards WHERE user_id = ? AND status = 'active'");
    $stmt->execute([$currentUserId]);
    $premiumCard = $stmt->fetch(PDO::FETCH_ASSOC);

    // Get user's support tickets
    $stmt = $pdo->prepare("SELECT * FROM support_tickets WHERE user_id = ? ORDER BY created_at DESC");
    $stmt->execute([$currentUserId]);
    $supportTickets = $stmt->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Support error: " . $e->getMessage());
    // Set sane defaults to prevent HTML errors
    $user = [
        'full_name' => 'User',
        'email' => 'your-email@example.com',
        'balance' => 0,
        'account_number' => 'Error',
        'account_type' => 'personal',
        'financial_account_type' => 'Error'
    ];
    $premiumCard = null;
    $supportTickets = [];
}

// Get user's first name
$firstName = 'User';
if (!empty($user['full_name']) && $user['full_name'] != 'User') {
    $nameParts = explode(' ', $user['full_name']);
    $firstName = $nameParts[0];
}

// Process support ticket submission
$ticketSuccess = false;
$ticketError = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_ticket'])) {
    $subject = $_POST['subject'] ?? '';
    $category = $_POST['category'] ?? '';
    $priority = $_POST['priority'] ?? '';
    $message = $_POST['message'] ?? '';
    
    // Validate form data
    if (empty($subject) || empty($category) || empty($priority) || empty($message)) {
        $ticketError = "Please fill in all required fields.";
    } else {
        try {
            $pdo = getDBConnection();
            
            // Insert ticket into database
            $stmt = $pdo->prepare("INSERT INTO support_tickets (user_id, subject, category, priority, message, status) 
                                   VALUES (?, ?, ?, ?, ?, 'open')");
            $stmt->execute([$currentUserId, $subject, $category, $priority, $message]);
            
            $ticketSuccess = true;
            
            // Send email notification to support
            $userEmail = $user['email'] ?? 'your-email@example.com';
            $userName = $user['full_name'] ?? 'User';
            $supportEmail = "support@airwavescreditsunion.com";
            $emailSubject = "New Support Ticket: $subject";
            $emailMessage = "
                A new support ticket has been submitted:\n\n
                User: $userName ($userEmail)\n
                Subject: $subject\n
                Category: $category\n
                Priority: $priority\n
                Message: $message\n\n
                Please log in to the support portal to respond.
            ";
            
            // Send email (you can enable this in production)
            // sendEmail($supportEmail, $emailSubject, $emailMessage);
            
        } catch (Exception $e) {
            error_log("Ticket submission error: " . $e->getMessage());
            $ticketError = "There was an error submitting your ticket. Please try again.";
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Support - AIRWAVES</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        /* Custom scrollbar */
        ::-webkit-scrollbar {
            width: 6px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb {
            background: #d1d5db;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb:hover {
            background: #9ca3af;
        }
        
        /* Card hover effects */
        .card-hover {
            transition: all 0.3s ease;
        }
        
        .card-hover:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        }
        
        /* Gradient cards */
        .gradient-card {
            background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 50%, #ffffff 100%);
            border: 1px solid #dcfce7;
        }
        
        /* Sidebar styling */
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        /* Premium card styling */
        .premium-card {
            background: linear-gradient(135deg, #09010f 0%, #facc15 100%);
        }
        
        /* Text shadow for better readability */
        .text-sharp {
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
        }
        
        /* Animations */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Form styling */
        .form-input {
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
        
        /* Status badges */
        .status-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .status-open {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        .status-pending {
            background-color: #fef3c7;
            color: #d97706;
        }
        
        .status-resolved {
            background-color: #dbeafe;
            color: #1e40af;
        }
        
        .status-closed {
            background-color: #e5e7eb;
            color: #374151;
        }
        
        /* Priority badges */
        .priority-badge {
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
            text-transform: capitalize;
        }
        
        .priority-low {
            background-color: #d1fae5;
            color: #065f46;
        }
        
        .priority-medium {
            background-color: #fef3c7;
            color: #d97706;
        }
        
        .priority-high {
            background-color: #fed7d7;
            color: #c53030;
        }
        
        .priority-urgent {
            background-color: #fed7d7;
            color: #c53030;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% {
                opacity: 1;
            }
            50% {
                opacity: 0.7;
            }
            100% {
                opacity: 1;
            }
        }
        
        /* Chat styling */
        .chat-container {
            height: 400px;
            overflow-y: auto;
        }
        
        .chat-message {
            max-width: 80%;
            margin-bottom: 1rem;
            padding: 0.75rem 1rem;
            border-radius: 1rem;
        }
        
        .chat-message.user {
            background-color: #22c55e;
            color: white;
            margin-left: auto;
            border-bottom-right-radius: 0.25rem;
        }
        
        .chat-message.support {
            background-color: #f3f4f6;
            color: #374151;
            margin-right: auto;
            border-bottom-left-radius: 0.25rem;
        }
        
        /* Loading spinner */
        .spinner {
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        
        /* Tab styling */
        .tab-button {
            transition: all 0.3s ease;
            border-radius: 12px;
        }
        
        .tab-button.active {
            background-color: #22c55e;
            color: white;
        }
        
        /* Support card animations */
        .support-card {
            transition: all 0.3s ease;
        }
        
        .support-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Hamburger Menu (Mobile) -->
                    <button id="sidebar-toggle" class="md:hidden mr-3 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                    </div>
                </div>
                
                <!-- User Info & Logout (Right Side) -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($user['full_name'] ?? 'U', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($user['full_name'] ?? 'User'); ?></span>
                    </div>
                    <a href="logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Left Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen pt-16 transition-transform -translate-x-full bg-white border-r border-gray-200 md:translate-x-0 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-home text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="wire_transfer.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-paper-plane text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Wire Transfer</span>
                    </a>
                </li>
                <li>
                    <a href="deposit.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-wallet text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Deposit</span>
                    </a>
                </li>
                <li>
                    <a href="transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-receipt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">History</span>
                    </a>
                </li>
                <li>
                    <a href="track_money.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-line text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Track Money</span>
                    </a>
                </li>
                <li>
                    <a href="add_recipient.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-user-plus text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Add Recipient</span>
                    </a>
                </li>
                 <li>
                    <a href="premiumcard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-gem text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Premium Card</span>
                    </a>
                </li>
                <li>
                    <a href="support.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-headset text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Support</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer with decorative element -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="gradient-card rounded-xl p-4 text-center">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Secure Banking</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main id="main-content" class="md:ml-64 pt-16 h-full pb-24 md:pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
        
            <!-- Welcome Section - Clear and bold -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-3xl font-bold text-gray-900 text-sharp">Customer Support 🛠️</h1>
                <p class="text-gray-600 mt-1 font-medium">We're here to help you 24/7. Choose your preferred support method.</p>
            </div>

            <!-- Support Options Tabs -->
            <div class="bg-white rounded-xl shadow-sm mb-8 animate-slide-up">
                <div class="border-b border-gray-200">
                    <nav class="flex flex-wrap -mb-px">
                        <button id="faq-tab" class="tab-button py-4 px-6 text-sm font-medium text-gray-500 hover:text-gray-700 active">
                            <i class="fas fa-question-circle mr-2"></i>FAQ & Help Center
                        </button>
                        <button id="ticket-tab" class="tab-button py-4 px-6 text-sm font-medium text-gray-500 hover:text-gray-700">
                            <i class="fas fa-ticket-alt mr-2"></i>Support Tickets
                        </button>
                        <button id="chat-tab" class="tab-button py-4 px-6 text-sm font-medium text-gray-500 hover:text-gray-700">
                            <i class="fas fa-comments mr-2"></i>Live Chat
                        </button>
                        <button id="contact-tab" class="tab-button py-4 px-6 text-sm font-medium text-gray-500 hover:text-gray-700">
                            <i class="fas fa-envelope mr-2"></i>Contact Us
                        </button>
                    </nav>
                </div>
            </div>

            <!-- FAQ & Help Center Tab Content -->
            <div id="faq-content" class="tab-content active animate-slide-up">
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    <!-- Main FAQ Content -->
                    <div class="lg:col-span-2">
                        <div class="gradient-card rounded-xl p-6 shadow-sm mb-6">
                            <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                                <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                                Frequently Asked Questions
                            </h2>
                            
                            <div class="space-y-4">
                                <!-- FAQ Item -->
                                <div class="bg-white border border-gray-200 rounded-xl overflow-hidden">
                                    <button class="faq-question w-full text-left p-4 font-medium text-gray-900 hover:bg-gray-50 flex justify-between items-center">
                                        <span>How long do wire transfers take to process?</span>
                                        <i class="fas fa-chevron-down text-gray-400 transition-transform"></i>
                                    </button>
                                    <div class="faq-answer p-4 bg-gray-50 border-t border-gray-200 hidden">
                                        <p class="text-gray-600">Domestic wire transfers typically process within 1-2 business days. International transfers may take 3-5 business days. You can track your transfer status in the "Track Money" section.</p>
                                    </div>
                                </div>
                                
                                <!-- FAQ Item -->
                                <div class="bg-white border border-gray-200 rounded-xl overflow-hidden">
                                    <button class="faq-question w-full text-left p-4 font-medium text-gray-900 hover:bg-gray-50 flex justify-between items-center">
                                        <span>What are the deposit limits for my account?</span>
                                        <i class="fas fa-chevron-down text-gray-400 transition-transform"></i>
                                    </button>
                                    <div class="faq-answer p-4 bg-gray-50 border-t border-gray-200 hidden">
                                        <p class="text-gray-600">Standard accounts have a daily deposit limit of $5,000 and a monthly limit of $25,000. Premium account holders enjoy higher limits - up to $50,000 daily and $250,000 monthly.</p>
                                    </div>
                                </div>
                                
                                <!-- FAQ Item -->
                                <div class="bg-white border border-gray-200 rounded-xl overflow-hidden">
                                    <button class="faq-question w-full text-left p-4 font-medium text-gray-900 hover:bg-gray-50 flex justify-between items-center">
                                        <span>How do I activate my premium card?</span>
                                        <i class="fas fa-chevron-down text-gray-400 transition-transform"></i>
                                    </button>
                                    <div class="faq-answer p-4 bg-gray-50 border-t border-gray-200 hidden">
                                        <p class="text-gray-600">Once your premium card application is approved, you can activate it in the "Premium Card" section. You'll need to set your PIN and confirm activation via OTP sent to your registered email.</p>
                                    </div>
                                </div>
                                
                                <!-- FAQ Item -->
                                <div class="bg-white border border-gray-200 rounded-xl overflow-hidden">
                                    <button class="faq-question w-full text-left p-4 font-medium text-gray-900 hover:bg-gray-50 flex justify-between items-center">
                                        <span>What should I do if I suspect fraudulent activity?</span>
                                        <i class="fas fa-chevron-down text-gray-400 transition-transform"></i>
                                    </button>
                                    <div class="faq-answer p-4 bg-gray-50 border-t border-gray-200 hidden">
                                        <p class="text-gray-600">Immediately contact our 24/7 support team via live chat or call our emergency hotline at 1-800-AIRWAVES-1. We recommend changing your password and reviewing recent transactions in your account.</p>
                                    </div>
                                </div>
                                
                                <!-- FAQ Item -->
                                <div class="bg-white border border-gray-200 rounded-xl overflow-hidden">
                                    <button class="faq-question w-full text-left p-4 font-medium text-gray-900 hover:bg-gray-50 flex justify-between items-center">
                                        <span>How can I increase my transfer limits?</span>
                                        <i class="fas fa-chevron-down text-gray-400 transition-transform"></i>
                                    </button>
                                    <div class="faq-answer p-4 bg-gray-50 border-t border-gray-200 hidden">
                                        <p class="text-gray-600">Transfer limits can be increased by upgrading to a premium account or by submitting a limit increase request through the "Profile & Settings" section. Additional verification may be required.</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Quick Help Cards -->
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div class="support-card bg-white rounded-xl p-6 border border-gray-200 shadow-sm">
                                <div class="w-12 h-12 rounded-full bg-blue-100 flex items-center justify-center mb-4">
                                    <i class="fas fa-file-invoice-dollar text-blue-600 text-xl"></i>
                                </div>
                                <h3 class="text-lg font-bold text-gray-900 mb-2">Account Statements</h3>
                                <p class="text-gray-600 text-sm mb-4">Download your monthly statements and transaction history.</p>
                                <button class="text-blue-600 hover:text-blue-800 font-medium text-sm flex items-center">
                                    Download Statements
                                    <i class="fas fa-arrow-right ml-1 text-xs"></i>
                                </button>
                            </div>
                            
                            <div class="support-card bg-white rounded-xl p-6 border border-gray-200 shadow-sm">
                                <div class="w-12 h-12 rounded-full bg-green-100 flex items-center justify-center mb-4">
                                    <i class="fas fa-shield-alt text-green-600 text-xl"></i>
                                </div>
                                <h3 class="text-lg font-bold text-gray-900 mb-2">Security Center</h3>
                                <p class="text-gray-600 text-sm mb-4">Manage your security settings and two-factor authentication.</p>
                                <button class="text-green-600 hover:text-green-800 font-medium text-sm flex items-center">
                                    Security Settings
                                    <i class="fas fa-arrow-right ml-1 text-xs"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Support Resources Sidebar -->
                    <div class="space-y-6">
                        <!-- Contact Card -->
                        <div class="gradient-card rounded-xl p-6 shadow-sm">
                            <h3 class="text-lg font-bold text-gray-900 mb-4 flex items-center">
                                <i class="fas fa-phone-alt text-green-600 mr-2"></i>
                                Contact Support
                            </h3>
                            <div class="space-y-3">
                                <div class="flex items-center p-3 bg-white rounded-lg border border-gray-200">
                                    <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-envelope text-green-600"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-semibold text-gray-900">Email</p>
                                        <p class="text-xs text-gray-500">support@airwavescreditsunion.com</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-center p-3 bg-white rounded-lg border border-gray-200">
                                    <div class="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-phone text-blue-600"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-semibold text-gray-900">Phone</p>
                                        <p class="text-xs text-gray-500">1-800-AIRWAVES-1</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-center p-3 bg-white rounded-lg border border-gray-200">
                                    <div class="w-10 h-10 rounded-full bg-purple-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-clock text-purple-600"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-semibold text-gray-900">Hours</p>
                                        <p class="text-xs text-gray-500">24/7 Support Available</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Emergency Support -->
                        <div class="bg-red-50 border border-red-200 rounded-xl p-6">
                            <div class="flex items-start">
                                <div class="w-10 h-10 rounded-full bg-red-100 flex items-center justify-center mr-3 flex-shrink-0">
                                    <i class="fas fa-exclamation-triangle text-red-600"></i>
                                </div>
                                <div>
                                    <h3 class="text-lg font-bold text-red-800 mb-2">Emergency Support</h3>
                                    <p class="text-red-700 text-sm mb-3">For lost/stolen cards or suspected fraud</p>
                                    <div class="space-y-2">
                                        <a href="tel:1-800-AIRWAVES-1" class="block w-full bg-red-600 hover:bg-red-700 text-white font-semibold py-2 px-4 rounded-lg text-center transition-colors">
                                            <i class="fas fa-phone mr-2"></i>Call Emergency Line
                                        </a>
                                        <button class="block w-full bg-white hover:bg-red-100 text-red-600 font-semibold py-2 px-4 rounded-lg border border-red-300 text-center transition-colors">
                                            <i class="fas fa-lock mr-2"></i>Freeze Account
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Support Tickets Tab Content -->
            <div id="ticket-content" class="tab-content hidden animate-slide-up">
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    <!-- Ticket List -->
                    <div class="lg:col-span-2">
                        <div class="gradient-card rounded-xl p-6 shadow-sm">
                            <div class="flex justify-between items-center mb-6">
                                <h2 class="text-xl font-bold text-gray-900 flex items-center">
                                    <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                                    My Support Tickets
                                </h2>
                                <button id="new-ticket-btn" class="bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-4 rounded-xl transition-colors duration-300 flex items-center">
                                    <i class="fas fa-plus mr-2"></i>New Ticket
                                </button>
                            </div>
                            
                            <?php if (!empty($supportTickets)): ?>
                                <div class="space-y-4">
                                    <?php foreach ($supportTickets as $ticket): ?>
                                        <div class="bg-white border border-gray-200 rounded-xl p-4 hover:shadow-md transition-shadow">
                                            <div class="flex justify-between items-start mb-2">
                                                <h3 class="font-semibold text-gray-900"><?php echo htmlspecialchars($ticket['subject']); ?></h3>
                                                <div class="flex space-x-2">
                                                    <span class="status-badge status-<?php echo $ticket['status']; ?>">
                                                        <?php echo ucfirst($ticket['status']); ?>
                                                    </span>
                                                    <span class="priority-badge priority-<?php echo strtolower($ticket['priority']); ?>">
                                                        <?php echo ucfirst($ticket['priority']); ?>
                                                    </span>
                                                </div>
                                            </div>
                                            <p class="text-gray-600 text-sm mb-3"><?php echo htmlspecialchars(substr($ticket['message'], 0, 150) . (strlen($ticket['message']) > 150 ? '...' : '')); ?></p>
                                            <div class="flex justify-between items-center text-xs text-gray-500">
                                                <span>Category: <?php echo htmlspecialchars($ticket['category']); ?></span>
                                                <span>Created: <?php echo date('M j, Y', strtotime($ticket['created_at'])); ?></span>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php else: ?>
                                <div class="text-center py-12">
                                    <div class="w-16 h-16 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                                        <i class="fas fa-ticket-alt text-gray-400 text-2xl"></i>
                                    </div>
                                    <p class="text-gray-500 font-medium">No support tickets yet</p>
                                    <p class="text-sm text-gray-400 mt-1">Create your first support ticket to get help</p>
                                    <button id="new-ticket-empty-btn" class="mt-4 bg-green-600 hover:bg-green-700 text-white font-semibold py-2 px-4 rounded-xl transition-colors duration-300">
                                        Create New Ticket
                                    </button>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- New Ticket Form -->
                    <div id="new-ticket-form" class="gradient-card rounded-xl p-6 shadow-sm hidden">
                        <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Create New Support Ticket
                        </h2>
                        
                        <?php if ($ticketSuccess): ?>
                            <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center">
                                <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center mr-4">
                                    <i class="fas fa-check text-green-600"></i>
                                </div>
                                <div>
                                    <p class="font-semibold text-green-800">Ticket Submitted Successfully!</p>
                                    <p class="text-sm text-green-700">We'll get back to you within 24 hours.</p>
                                </div>
                            </div>
                        <?php elseif ($ticketError): ?>
                            <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center">
                                <div class="w-10 h-10 rounded-full bg-red-100 flex items-center justify-center mr-4">
                                    <i class="fas fa-exclamation-triangle text-red-600"></i>
                                </div>
                                <div>
                                    <p class="font-semibold text-red-800">Submission Failed</p>
                                    <p class="text-sm text-red-700"><?php echo htmlspecialchars($ticketError); ?></p>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" action="support.php">
                            <div class="space-y-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Subject *</label>
                                    <input type="text" name="subject" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" placeholder="Brief description of your issue" required>
                                </div>
                                
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Category *</label>
                                        <select name="category" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" required>
                                            <option value="">Select a category</option>
                                            <option value="account">Account Issues</option>
                                            <option value="transfer">Wire Transfers</option>
                                            <option value="deposit">Deposits</option>
                                            <option value="card">Premium Card</option>
                                            <option value="security">Security Concern</option>
                                            <option value="technical">Technical Issue</option>
                                            <option value="other">Other</option>
                                        </select>
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Priority *</label>
                                        <select name="priority" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" required>
                                            <option value="">Select priority</option>
                                            <option value="low">Low</option>
                                            <option value="medium">Medium</option>
                                            <option value="high">High</option>
                                            <option value="urgent">Urgent</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Message *</label>
                                    <textarea name="message" rows="6" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" placeholder="Please describe your issue in detail..." required></textarea>
                                </div>
                                
                                <div class="pt-4">
                                    <button type="submit" name="submit_ticket" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-3 px-4 rounded-xl transition-colors duration-300 flex items-center justify-center">
                                        <i class="fas fa-paper-plane mr-2"></i>
                                        Submit Ticket
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Live Chat Tab Content -->
            <div id="chat-content" class="tab-content hidden animate-slide-up">
                <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    <!-- Chat Interface -->
                    <div class="lg:col-span-2">
                        <div class="gradient-card rounded-xl p-6 shadow-sm">
                            <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                                <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                                Live Chat Support
                            </h2>
                            
                            <div class="bg-white border border-gray-200 rounded-xl p-4 mb-4">
                                <div class="flex items-center justify-between mb-4">
                                    <div class="flex items-center">
                                        <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center mr-3">
                                            <i class="fas fa-headset text-green-600"></i>
                                        </div>
                                        <div>
                                            <p class="font-semibold text-gray-900">Airwaves Support</p>
                                            <p class="text-xs text-gray-500">Online - Typically replies in 2 minutes</p>
                                        </div>
                                    </div>
                                    <div class="flex items-center">
                                        <span class="w-3 h-3 bg-green-500 rounded-full mr-2"></span>
                                        <span class="text-sm text-gray-600">Online</span>
                                    </div>
                                </div>
                                
                                <!-- Chat Messages -->
                                <div class="chat-container border border-gray-200 rounded-lg p-4 bg-gray-50 mb-4">
                                    <div class="chat-message support">
                                        <p class="font-medium">Hello <?php echo $firstName; ?>! 👋</p>
                                        <p class="text-sm">Welcome to Airwaves Credit Union support. How can we help you today?</p>
                                        <span class="text-xs text-gray-500 block mt-1">2:30 PM</span>
                                    </div>
                                    
                                    <!-- User messages will be appended here -->
                                </div>
                                
                                <!-- Chat Input -->
                                <div class="flex space-x-2">
                                    <input type="text" id="chat-input" placeholder="Type your message..." class="form-input flex-1 rounded-xl border border-gray-300 px-4 py-3 focus:outline-none">
                                    <button id="send-chat-btn" class="bg-green-600 hover:bg-green-700 text-white font-semibold py-3 px-6 rounded-xl transition-colors duration-300">
                                        <i class="fas fa-paper-plane"></i>
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Chat Help -->
                    <div class="space-y-6">
                        <div class="gradient-card rounded-xl p-6 shadow-sm">
                            <h3 class="text-lg font-bold text-gray-900 mb-4 flex items-center">
                                <i class="fas fa-lightbulb text-yellow-500 mr-2"></i>
                                Chat Tips
                            </h3>
                            <ul class="space-y-2 text-sm text-gray-600">
                                <li class="flex items-start">
                                    <i class="fas fa-check text-green-500 mr-2 mt-1 flex-shrink-0"></i>
                                    <span>Have your account information ready</span>
                                </li>
                                <li class="flex items-start">
                                    <i class="fas fa-check text-green-500 mr-2 mt-1 flex-shrink-0"></i>
                                    <span>Describe your issue clearly and concisely</span>
                                </li>
                                <li class="flex items-start">
                                    <i class="fas fa-check text-green-500 mr-2 mt-1 flex-shrink-0"></i>
                                    <span>Be ready to verify your identity</span>
                                </li>
                                <li class="flex items-start">
                                    <i class="fas fa-check text-green-500 mr-2 mt-1 flex-shrink-0"></i>
                                    <span>Save the chat transcript for your records</span>
                                </li>
                            </ul>
                        </div>
                        
                        <div class="bg-blue-50 border border-blue-200 rounded-xl p-6">
                            <div class="flex items-start">
                                <div class="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center mr-3 flex-shrink-0">
                                    <i class="fas fa-robot text-blue-600"></i>
                                </div>
                                <div>
                                    <h3 class="text-lg font-bold text-blue-800 mb-2">AI Assistant</h3>
                                    <p class="text-blue-700 text-sm mb-3">Get instant answers to common questions</p>
                                    <button class="w-full bg-blue-600 hover:bg-blue-700 text-white font-semibold py-2 px-4 rounded-lg transition-colors flex items-center justify-center">
                                        <i class="fas fa-bolt mr-2"></i>Try AI Assistant
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Contact Us Tab Content -->
            <div id="contact-content" class="tab-content hidden animate-slide-up">
                <div class="grid grid-cols-1 lg:grid-cols-2 gap-8">
                    <!-- Contact Information -->
                    <div class="space-y-6">
                        <div class="gradient-card rounded-xl p-6 shadow-sm">
                            <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                                <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                                Get In Touch
                            </h2>
                            
                            <div class="space-y-4">
                                <div class="flex items-center p-4 bg-white rounded-xl border border-gray-200">
                                    <div class="w-12 h-12 rounded-full bg-green-100 flex items-center justify-center mr-4">
                                        <i class="fas fa-envelope text-green-600 text-xl"></i>
                                    </div>
                                    <div>
                                        <p class="font-semibold text-gray-900">Email Us</p>
                                        <p class="text-gray-600">support@airwavescreditsunion.com</p>
                                        <p class="text-sm text-gray-500 mt-1">Typically replies within 2 hours</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-center p-4 bg-white rounded-xl border border-gray-200">
                                    <div class="w-12 h-12 rounded-full bg-blue-100 flex items-center justify-center mr-4">
                                        <i class="fas fa-phone text-blue-600 text-xl"></i>
                                    </div>
                                    <div>
                                        <p class="font-semibold text-gray-900">Call Us</p>
                                        <p class="text-gray-600">1-800-AIRWAVES-1</p>
                                        <p class="text-sm text-gray-500 mt-1">24/7 customer support</p>
                                    </div>
                                </div>
                                
                                <div class="flex items-center p-4 bg-white rounded-xl border border-gray-200">
                                    <div class="w-12 h-12 rounded-full bg-purple-100 flex items-center justify-center mr-4">
                                        <i class="fas fa-map-marker-alt text-purple-600 text-xl"></i>
                                    </div>
                                    <div>
                                        <p class="font-semibold text-gray-900">Visit Us</p>
                                        <p class="text-gray-600">123 Financial District</p>
                                        <p class="text-gray-600">New York, NY 10005</p>
                                        <p class="text-sm text-gray-500 mt-1">Mon-Fri 9:00 AM - 5:00 PM EST</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Social Media -->
                        <div class="gradient-card rounded-xl p-6 shadow-sm">
                            <h3 class="text-lg font-bold text-gray-900 mb-4">Follow Us</h3>
                            <div class="flex space-x-4">
                                <a href="#" class="w-10 h-10 rounded-full bg-blue-100 flex items-center justify-center text-blue-600 hover:bg-blue-200 transition-colors">
                                    <i class="fab fa-twitter"></i>
                                </a>
                                <a href="#" class="w-10 h-10 rounded-full bg-blue-600 flex items-center justify-center text-white hover:bg-blue-700 transition-colors">
                                    <i class="fab fa-facebook-f"></i>
                                </a>
                                <a href="#" class="w-10 h-10 rounded-full bg-pink-100 flex items-center justify-center text-pink-600 hover:bg-pink-200 transition-colors">
                                    <i class="fab fa-instagram"></i>
                                </a>
                                <a href="#" class="w-10 h-10 rounded-full bg-blue-700 flex items-center justify-center text-white hover:bg-blue-800 transition-colors">
                                    <i class="fab fa-linkedin-in"></i>
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Contact Form -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm">
                        <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Send Us a Message
                        </h2>
                        
                        <form class="space-y-4">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">First Name</label>
                                    <input type="text" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" value="<?php echo htmlspecialchars($firstName); ?>" readonly>
                                </div>
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Last Name</label>
                                    <input type="text" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" value="<?php echo htmlspecialchars(isset($nameParts[1]) ? $nameParts[1] : ''); ?>" readonly>
                                </div>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Email</label>
                                <input type="email" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" value="<?php echo htmlspecialchars($user['email'] ?? ''); ?>" readonly>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Subject</label>
                                <input type="text" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" placeholder="What is this regarding?">
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray-700 mb-2">Message</label>
                                <textarea rows="5" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none" placeholder="How can we help you?"></textarea>
                            </div>
                            
                            <div class="pt-4">
                                <button type="submit" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-3 px-4 rounded-xl transition-colors duration-300 flex items-center justify-center">
                                    <i class="fas fa-paper-plane mr-2"></i>
                                    Send Message
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Mobile Navigation -->
    <div class="fixed bottom-0 left-0 right-0 md:hidden z-50">
        <div class="bg-white rounded-t-2xl shadow-lg border-t border-gray-200">
            <div class="flex justify-around items-center py-3">
                <a href="dashboard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-home text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Dashboard</span>
                </a>
                <a href="wire_transfer.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-paper-plane text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Transfer</span>
                </a>
                <a href="deposit.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-wallet text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Deposit</span>
                </a>
                <a href="track_money.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-chart-line text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Track</span>
                </a>
                <a href="support.php" class="flex flex-col items-center text-green-600">
                    <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mb-1">
                         <i class="fas fa-headset text-lg"></i>
                    </div>
                    <span class="text-xs font-semibold">Support</span>
                </a>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebar-toggle');
            const sidebar = document.getElementById('sidebar');

            if (sidebarToggle && sidebar) {
                sidebarToggle.addEventListener('click', function(e) {
                    e.stopPropagation();
                    sidebar.classList.toggle('-translate-x-full');
                    sidebar.classList.toggle('translate-x-0');
                });

                document.addEventListener('click', function(e) {
                    if (sidebar.classList.contains('translate-x-0') && !sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        sidebar.classList.add('-translate-x-full');
                        sidebar.classList.remove('translate-x-0');
                    }
                });
            }
            
            // Tab Switching Functionality
            const tabButtons = document.querySelectorAll('.tab-button');
            const tabContents = document.querySelectorAll('.tab-content');
            
            tabButtons.forEach(button => {
                button.addEventListener('click', function() {
                    // Remove active class from all tabs
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    tabContents.forEach(content => content.classList.remove('active'));
                    
                    // Add active class to clicked tab
                    this.classList.add('active');
                    
                    // Show corresponding content
                    const tabId = this.id.replace('-tab', '-content');
                    document.getElementById(tabId).classList.add('active');
                });
            });
            
            // FAQ Accordion
            const faqQuestions = document.querySelectorAll('.faq-question');
            
            faqQuestions.forEach(question => {
                question.addEventListener('click', function() {
                    const answer = this.nextElementSibling;
                    const icon = this.querySelector('i');
                    
                    // Toggle answer visibility
                    answer.classList.toggle('hidden');
                    
                    // Rotate icon
                    icon.classList.toggle('fa-chevron-down');
                    icon.classList.toggle('fa-chevron-up');
                });
            });
            
            // Support Ticket Form Toggle
            const newTicketBtn = document.getElementById('new-ticket-btn');
            const newTicketEmptyBtn = document.getElementById('new-ticket-empty-btn');
            const newTicketForm = document.getElementById('new-ticket-form');
            const ticketContent = document.getElementById('ticket-content');
            
            if (newTicketBtn) {
                newTicketBtn.addEventListener('click', function() {
                    newTicketForm.classList.toggle('hidden');
                    if (!newTicketForm.classList.contains('hidden')) {
                        newTicketForm.scrollIntoView({ behavior: 'smooth' });
                    }
                });
            }
            
            if (newTicketEmptyBtn) {
                newTicketEmptyBtn.addEventListener('click', function() {
                    // Switch to ticket tab and show form
                    tabButtons.forEach(btn => btn.classList.remove('active'));
                    tabContents.forEach(content => content.classList.remove('active'));
                    
                    document.getElementById('ticket-tab').classList.add('active');
                    document.getElementById('ticket-content').classList.add('active');
                    
                    newTicketForm.classList.remove('hidden');
                    newTicketForm.scrollIntoView({ behavior: 'smooth' });
                });
            }
            
            // Live Chat Functionality
            const chatInput = document.getElementById('chat-input');
            const sendChatBtn = document.getElementById('send-chat-btn');
            const chatContainer = document.querySelector('.chat-container');
            
            if (sendChatBtn && chatInput) {
                sendChatBtn.addEventListener('click', sendMessage);
                chatInput.addEventListener('keypress', function(e) {
                    if (e.key === 'Enter') {
                        sendMessage();
                    }
                });
            }
            
            function sendMessage() {
                const message = chatInput.value.trim();
                if (message) {
                    // Add user message
                    const userMessage = document.createElement('div');
                    userMessage.className = 'chat-message user';
                    userMessage.innerHTML = `
                        <p>${message}</p>
                        <span class="text-xs text-green-100 block mt-1">Just now</span>
                    `;
                    chatContainer.appendChild(userMessage);
                    
                    // Clear input
                    chatInput.value = '';
                    
                    // Scroll to bottom
                    chatContainer.scrollTop = chatContainer.scrollHeight;
                    
                    // Simulate support response after delay
                    setTimeout(() => {
                        const supportMessage = document.createElement('div');
                        supportMessage.className = 'chat-message support';
                        supportMessage.innerHTML = `
                            <p>Thank you for your message. One of our support agents will be with you shortly. In the meantime, is there any specific information you'd like to share about your issue?</p>
                            <span class="text-xs text-gray-500 block mt-1">Just now</span>
                        `;
                        chatContainer.appendChild(supportMessage);
                        
                        // Scroll to bottom again
                        chatContainer.scrollTop = chatContainer.scrollHeight;
                    }, 2000);
                }
            }
        });
    </script>

</body>
</html>