<?php
require_once 'config.php';

$errors = [];
$success = false;

$email = $_GET['email'] ?? $_POST['email'] ?? '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = sanitizeInput($_POST['email'] ?? '');
    
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Please enter a valid email address.";
    } else {
        try {
            $pdo = getDBConnection();
            
            // Check if user exists and isn't already verified
            $stmt = $pdo->prepare("SELECT id, full_name, email_verified FROM users WHERE email = ?");
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$user) {
                $errors[] = "No account found with this email address.";
            } elseif ($user['email_verified']) {
                $errors[] = "This email address is already verified.";
            } else {
                // Check if we've sent too many verification emails recently
                $stmt = $pdo->prepare("SELECT COUNT(*) FROM verification_tokens WHERE user_id = ? AND token_type = 'email_verification' AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)");
                $stmt->execute([$user['id']]);
                $recent_attempts = $stmt->fetchColumn();
                
                if ($recent_attempts >= 3) {
                    $errors[] = "Too many verification requests. Please try again in 1 hour.";
                } else {
                    // Generate new verification token
                    $verification_token = generateVerificationToken($user['id'], 'email_verification');
                    $verification_url = BASE_URL . "/verify-email.php?token=" . $verification_token;
                    
                    // Send verification email
                    $email_sent = sendVerificationEmail($email, $user['full_name'], $verification_url);
                    
                    if ($email_sent) {
                        $success = true;
                        
                        // Update verification sent time
                        $stmt = $pdo->prepare("UPDATE users SET verification_sent_at = NOW() WHERE id = ?");
                        $stmt->execute([$user['id']]);
                    } else {
                        $errors[] = "Failed to send verification email. Please try again.";
                    }
                }
            }
        } catch (PDOException $e) {
            $errors[] = "An error occurred. Please try again.";
            error_log("Resend verification error: " . $e->getMessage());
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Resend Verification - Oxland Credits</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-50 min-h-screen flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
    <div class="max-w-md w-full space-y-8">
        <div class="text-center">
            <span class="text-3xl font-black text-gray-900">OXLAND</span><span class="text-3xl font-black text-green-800">.</span>
        </div>

        <?php if ($success): ?>
            <div class="bg-white p-8 rounded-2xl shadow-lg border border-gray-200 text-center">
                <div class="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <i class="fas fa-check-circle text-green-600 text-2xl"></i>
                </div>
                
                <h2 class="text-2xl font-bold text-gray-900 mb-4">Verification Email Sent!</h2>
                
                <p class="text-gray-600 mb-6">
                    We've sent a new verification link to:<br>
                    <strong class="text-green-800"><?php echo htmlspecialchars($email); ?></strong>
                </p>
                
                <a href="login.php" 
                   class="w-full inline-flex justify-center items-center px-4 py-3 border border-transparent text-sm font-medium rounded-lg text-white bg-green-800 hover:bg-green-700 transition-colors duration-300">
                    <i class="fas fa-sign-in-alt mr-2"></i>
                    Return to Login
                </a>
            </div>
            
        <?php else: ?>
            <div class="bg-white p-8 rounded-2xl shadow-lg border border-gray-200">
                <h2 class="text-2xl font-bold text-gray-900 mb-6 text-center">Resend Verification Email</h2>
                
                <?php if (!empty($errors)): ?>
                    <div class="bg-red-50 border border-red-200 rounded-lg p-4 mb-6">
                        <div class="flex">
                            <div class="flex-shrink-0">
                                <i class="fas fa-exclamation-circle text-red-400"></i>
                            </div>
                            <div class="ml-3">
                                <ul class="list-disc list-inside text-sm text-red-700 space-y-1">
                                    <?php foreach ($errors as $error): ?>
                                        <li><?php echo htmlspecialchars($error); ?></li>
                                    <?php endforeach; ?>
                                </ul>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>

                <form method="POST" class="space-y-6">
                    <div>
                        <label for="email" class="block text-sm font-medium text-gray-700">Email Address</label>
                        <input id="email" name="email" type="email" required 
                               class="mt-1 appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-800 focus:border-transparent"
                               placeholder="Enter your email address"
                               value="<?php echo htmlspecialchars($email); ?>">
                    </div>

                    <button type="submit" 
                            class="w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-lg text-white bg-green-800 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-800 transition-colors duration-300">
                        <i class="fas fa-redo mr-2"></i>
                        Resend Verification Email
                    </button>
                </form>

                <div class="mt-6 text-center">
                    <p class="text-sm text-gray-600">
                        Remember your password? 
                        <a href="login.php" class="font-medium text-green-800 hover:text-green-700">Sign in here</a>
                    </p>
                </div>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>