<?php
require_once 'config.php';

$errors = [];

// Check if user is already logged in
if (isLoggedIn()) {
    redirect('dashboard.php');
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate CSRF token
    if (!validateCSRFToken($_POST['csrf_token'] ?? '')) {
        $errors[] = "Security token invalid. Please try again.";
    }
    
    // Sanitize inputs
    $email = sanitizeInput($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $remember = isset($_POST['remember']);
    
    // Validate inputs
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Valid email address is required.";
    }
    
    if (empty($password)) {
        $errors[] = "Password is required.";
    }
    
    // If no errors, attempt login
    if (empty($errors)) {
        try {
            $pdo = getDBConnection();
            
            // Check login attempts (basic rate limiting)
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM login_attempts WHERE email = ? AND attempt_time > DATE_SUB(NOW(), INTERVAL 15 MINUTE) AND success = 0");
            $stmt->execute([$email]);
            $recent_failures = $stmt->fetchColumn();
            
            if ($recent_failures >= 5) {
                $errors[] = "Too many failed login attempts. Please try again in 15 minutes.";
            } else {
                // Get user by email
                $stmt = $pdo->prepare("SELECT id, email, password, full_name, is_active FROM users WHERE email = ?");
                $stmt->execute([$email]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Log login attempt
                $stmt = $pdo->prepare("INSERT INTO login_attempts (email, ip_address, success) VALUES (?, ?, ?)");
                $ip_address = $_SERVER['REMOTE_ADDR'];
                
                if ($user && password_verify($password, $user['password'])) {
                    if (!$user['is_active']) {
                        $errors[] = "Your account has been deactivated. Please contact support.";
                        $stmt->execute([$email, $ip_address, false]);
                    } else {
                        // Successful PASSWORD validation, now send OTP
                        
                        // Log the attempt
                        $stmt->execute([$email, $ip_address, true]);
                        
                        // --- NEW OTP LOGIC ---
                        // Instead of logging in, send an OTP
                        $otp_sent = sendLoginOTP($user['id'], $user['email'], $user['full_name']);
                        
                        if ($otp_sent) {
                            // OTP email was sent.
                            // Store user ID temporarily in session for verification page
                            $_SESSION['otp_user_id'] = $user['id'];
                            $_SESSION['otp_user_email'] = $user['email']; // To display on the OTP page
                            
                            // Redirect to the new OTP entry page
                            redirect('verify-otp.php');
                            exit;
                        } else {
                            // Email sending failed
                            $errors[] = "We couldn't send your login OTP. Please try again later.";
                            // This will show the error on the login page
                        }
                        // --- END OF NEW OTP LOGIC ---
                    }
                } else {
                    // Failed login
                    $stmt->execute([$email, $ip_address, false]);
                    $errors[] = "Invalid email or password.";
                }
            }
        } catch (PDOException $e) {
            $errors[] = "Login failed. Please try again.";
            error_log("Login error: " . $e->getMessage());
        }
    }
}

// Check for registration success
$registration_success = false;
if (isset($_SESSION['registration_success'])) {
    $registration_success = true;
    unset($_SESSION['registration_success']);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - Airwaves Credit Union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;600;700;800;900&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-blue: #0d3b5d;
            --secondary-blue: #1e4e75;
            --accent-blue: #2c5f8a;
        }
        
        body { 
            font-family: 'Inter', sans-serif; 
            background: linear-gradient(135deg, #0d3b5d 0%, #1e4e75 50%, #2c5f8a 100%);
            min-height: 100vh;
            position: relative;
            overflow-x: hidden;
        }
        
        body::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-image: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1200 800" opacity="0.03"><path fill="%23ffffff" d="M0,0H1200V800H0Z"/><circle cx="200" cy="200" r="150" fill="%23ffffff"/><circle cx="800" cy="600" r="120" fill="%23ffffff"/><rect x="400" y="100" width="300" height="200" fill="%23ffffff"/></svg>');
            background-size: cover;
            background-position: center;
            pointer-events: none;
        }
        
        .glass-effect {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(20px);
            border: 1px solid rgba(255, 255, 255, 0.2);
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
        }
        
        .floating-element {
            animation: float 6s ease-in-out infinite;
        }
        
        @keyframes float {
            0%, 100% { transform: translateY(0px); }
            50% { transform: translateY(-20px); }
        }
        
        .senior-illustration {
            background-image: url('data:image/svg+xml,<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 200 200" opacity="0.1"><path fill="%23ffffff" d="M100,50c27.61,0,50,22.39,50,50s-22.39,50-50,50S50,127.61,50,100,72.39,50,100,50Zm0,10c-22.09,0-40,17.91-40,40s17.91,40,40,40,40-17.91,40-40S122.09,60,100,60Z"/><circle cx="85" cy="85" r="5" fill="%23ffffff"/><circle cx="115" cy="85" r="5" fill="%23ffffff"/><path fill="%23ffffff" d="M80,115c0-11.05,8.95-20,20-20s20,8.95,20,20Z"/></svg>');
            background-size: contain;
            background-repeat: no-repeat;
            background-position: center;
        }
        
        .logo-circle {
            width: 120px;
            height: 120px;
            background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1), 0 0 0 1px rgba(255, 255, 255, 0.2);
            border: 4px solid rgba(255, 255, 255, 0.3);
        }
        
        .logo-image {
            width: 70px;
            height: 70px;
            object-fit: contain;
            filter: drop-shadow(0 4px 8px rgba(0, 0, 0, 0.2));
        }
    </style>
</head>
<body class="flex items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
    <!-- Background Elements -->
    <div class="absolute inset-0 overflow-hidden">
        <div class="absolute -top-40 -right-32 w-80 h-80 bg-white rounded-full opacity-10 floating-element"></div>
        <div class="absolute top-1/2 -left-20 w-60 h-60 bg-white rounded-full opacity-10 floating-element" style="animation-delay: -2s;"></div>
        <div class="absolute bottom-20 right-1/4 w-40 h-40 bg-white rounded-full opacity-10 floating-element" style="animation-delay: -4s;"></div>
        
        <!-- Senior Illustration Elements -->
        <div class="absolute top-20 left-10 w-32 h-32 senior-illustration opacity-20 floating-element"></div>
        <div class="absolute bottom-32 right-20 w-24 h-24 senior-illustration opacity-20 floating-element" style="animation-delay: -3s;"></div>
        <div class="absolute top-1/3 right-1/4 w-28 h-28 senior-illustration opacity-20 floating-element" style="animation-delay: -1.5s;"></div>
    </div>

    <div class="max-w-md w-full space-y-8 relative z-10">
        <!-- Header -->
        <div class="text-center">
            <!-- Logo with Circular Background -->
            <div class="flex justify-center mb-6">
                <div class="logo-circle floating-element">
                    <img src="Uploads/images/logo.png" alt="Airwaves Credit Union" class="logo-image">
                </div>
            </div>
            <h2 class="text-center text-3xl font-extrabold text-white">
                Welcome Back
            </h2>
            <p class="mt-2 text-center text-lg text-blue-100">
                Secure access to your financial world
            </p>
        </div>

        <!-- Success Message (after registration) -->
        <?php if ($registration_success): ?>
            <div class="glass-effect rounded-2xl p-4 border-l-4 border-green-500">
                <div class="flex items-center">
                    <div class="flex-shrink-0">
                        <i class="fas fa-check-circle text-green-500 text-lg"></i>
                    </div>
                    <div class="ml-3">
                        <p class="text-sm font-medium text-gray-800">
                            Registration successful! Please sign in with your credentials.
                        </p>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Error Messages -->
        <?php if (!empty($errors)): ?>
            <div class="glass-effect rounded-2xl p-4 border-l-4 border-red-500">
                <div class="flex">
                    <div class="flex-shrink-0">
                        <i class="fas fa-exclamation-circle text-red-500 text-lg"></i>
                    </div>
                    <div class="ml-3">
                        <h3 class="text-sm font-medium text-gray-800">
                            There were problems with your login:
                        </h3>
                        <div class="mt-2 text-sm text-gray-700">
                            <ul class="list-disc list-inside space-y-1">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        <?php endif; ?>

        <!-- Login Form -->
        <form class="glass-effect rounded-3xl p-8 space-y-6" method="POST" action="">
            <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">
            
            <div class="space-y-5">
                <!-- Email -->
                <div>
                    <label for="email" class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-envelope text-[#0d3b5d] mr-2"></i>
                        Email Address
                    </label>
                    <div class="relative">
                        <input id="email" name="email" type="email" required 
                               class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-[#0d3b5d] focus:border-transparent transition-all duration-300 bg-white/80"
                               placeholder="Enter your email"
                               value="<?php echo htmlspecialchars($_POST['email'] ?? ''); ?>">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <i class="fas fa-user text-gray-400"></i>
                        </div>
                    </div>
                </div>

                <!-- Password -->
                <div>
                    <label for="password" class="block text-sm font-semibold text-gray-700 mb-2">
                        <i class="fas fa-lock text-[#0d3b5d] mr-2"></i>
                        Password
                    </label>
                    <div class="relative">
                        <input id="password" name="password" type="password" required 
                               class="w-full px-4 py-3 pl-12 border border-gray-300 rounded-xl focus:outline-none focus:ring-2 focus:ring-[#0d3b5d] focus:border-transparent transition-all duration-300 bg-white/80"
                               placeholder="Enter your password">
                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                            <i class="fas fa-key text-gray-400"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Remember Me & Forgot Password -->
            <div class="flex items-center justify-between">
                <div class="flex items-center">
                    <input id="remember" name="remember" type="checkbox"
                           class="h-4 w-4 text-[#0d3b5d] focus:ring-[#0d3b5d] border-gray-300 rounded">
                    <label for="remember" class="ml-2 block text-sm text-gray-700 font-medium">
                        Remember me
                    </label>
                </div>

                <div class="text-sm">
                    <a href="forgot-password.php" class="font-semibold text-[#0d3b5d] hover:text-blue-800 transition-colors duration-300">
                        Forgot password?
                    </a>
                </div>
            </div>

            <!-- Submit Button -->
            <div>
                <button type="submit" 
                        class="group relative w-full flex justify-center py-4 px-4 border border-transparent text-lg font-semibold rounded-xl text-white bg-[#0d3b5d] hover:bg-blue-800 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-[#0d3b5d] transition-all duration-300 transform hover:scale-[1.02] shadow-lg">
                    <span class="absolute left-0 inset-y-0 flex items-center pl-4">
                        <i class="fas fa-sign-in-alt text-blue-200 group-hover:text-white transition-colors duration-300"></i>
                    </span>
                    Sign in to Dashboard
                </button>
            </div>

            <!-- Home Button -->
            <div class="text-center">
                <a href="https://airwavescreditsunion.com" 
                   class="inline-flex items-center space-x-2 px-6 py-3 bg-gradient-to-r from-green-500 to-green-600 hover:from-green-600 hover:to-green-700 text-white font-semibold rounded-xl transition-all duration-300 transform hover:scale-[1.02] shadow-lg">
                    <i class="fas fa-home"></i>
                    <span>Back to Home</span>
                </a>
            </div>

            <!-- Register Link -->
            <div class="text-center pt-4">
                <p class="text-sm text-gray-600 font-medium">
                    New to Airwaves?
                    <a href="register.php" class="font-bold text-[#0d3b5d] hover:text-blue-800 transition-colors duration-300 ml-1">
                        Create your account
                    </a>
                </p>
            </div>
        </form>

        <!-- Security Notice -->
        <div class="text-center">
            <div class="inline-flex items-center space-x-2 text-xs text-blue-200 bg-blue-900/30 px-4 py-2 rounded-full">
                <i class="fas fa-shield-alt"></i>
                <span>Bank-level security & 256-bit encryption</span>
            </div>
        </div>

        <!-- Senior-Friendly Features -->
        <div class="glass-effect rounded-2xl p-6 mt-6">
            <div class="flex items-center justify-between">
                <div class="flex items-center space-x-3">
                    <div class="w-10 h-10 bg-[#0d3b5d] rounded-full flex items-center justify-center">
                        <i class="fas fa-heart text-white text-sm"></i>
                    </div>
                    <div>
                        <p class="text-sm font-semibold text-gray-800">Senior-Friendly</p>
                        <p class="text-xs text-gray-600">Easy access for all generations</p>
                    </div>
                </div>
                <div class="text-2xl">👵👴</div>
            </div>
        </div>
    </div>

    <script>
        // Enhanced form interactions
        document.addEventListener('DOMContentLoaded', function() {
            const emailInput = document.getElementById('email');
            const passwordInput = document.getElementById('password');
            
            if (emailInput) {
                emailInput.focus();
                
                // Add input animations
                emailInput.addEventListener('focus', function() {
                    this.parentElement.classList.add('ring-2', 'ring-[#0d3b5d]', 'ring-opacity-20');
                });
                
                emailInput.addEventListener('blur', function() {
                    this.parentElement.classList.remove('ring-2', 'ring-[#0d3b5d]', 'ring-opacity-20');
                });
            }
            
            if (passwordInput) {
                passwordInput.addEventListener('focus', function() {
                    this.parentElement.classList.add('ring-2', 'ring-[#0d3b5d]', 'ring-opacity-20');
                });
                
                passwordInput.addEventListener('blur', function() {
                    this.parentElement.classList.remove('ring-2', 'ring-[#0d3b5d]', 'ring-opacity-20');
                });
            }
            
            // Add subtle background animation
            const floatingElements = document.querySelectorAll('.floating-element');
            floatingElements.forEach((element, index) => {
                element.style.animationDelay = `${-index * 2}s`;
            });
        });
    </script>
</body>
</html>