<?php
// --- REAL OTP LOGIC ---
// session_start() MUST be the very first thing to run
session_start();
require_once 'config.php';

// --- AJAX API HANDLER ---
// This block catches background requests from our JavaScript
if (isset($_POST['action']) && $_POST['action'] === 'send_deposit_otp') {
    // We are sending JSON back
    header('Content-Type: application/json');
    
    // Get user email (we need this to send the email)
    $currentUserId = getCurrentUserId();
    $pdo = getDBConnection();
    $stmt = $pdo->prepare("SELECT email, balance FROM users u LEFT JOIN accounts a ON u.id = a.user_id WHERE u.id = ? LIMIT 1");
    $stmt->execute([$currentUserId]);
    $userAccount = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$userAccount) {
        echo json_encode(['status' => 'error', 'message' => 'User not found.']);
        exit;
    }

    // Get form data from the AJAX request
    $depositType = $_POST['deposit_type'] ?? '';
    $amount = $_POST['amount'] ?? '';
    $referenceNumber = $_POST['reference_number'] ?? '';
    
    // Server-side validation
    if (empty($depositType) || empty($amount)) {
        echo json_encode(['status' => 'error', 'message' => 'Deposit type and amount are required.']);
        exit;
    }
    if (!is_numeric($amount) || $amount <= 0) {
        echo json_encode(['status' => 'error', 'message' => 'Please enter a valid amount.']);
        exit;
    }

    // --- OTP Generation & Email ---
    $otp = random_int(100000, 999999);
    // Store the OTP in the user's session, NOT in the client
    $_SESSION['deposit_otp'] = $otp;
    $_SESSION['deposit_otp_expiry'] = time() + 300; // OTP expires in 5 minutes

    $userEmail = $userAccount['email'] ?? 'your-email@example.com';
    $subject = "Your Airwaves Credit Union Deposit OTP";
    $message = "Your one-time password to confirm your deposit is: $otp\n\nThis code will expire in 5 minutes.";

    // Attempt to send the email
    if (sendEmail($userEmail, $subject, $message)) {
        // Email sent successfully
        echo json_encode(['status' => 'success', 'message' => 'OTP sent to ' . htmlspecialchars($userEmail)]);
    } else {
        // Mail failed (common on local servers)
        echo json_encode([
            'status' => 'error',
            'message' => 'Failed to send OTP email. Please contact support.',
        ]);
    }
    exit; // Stop script execution, we only want to return JSON
}
// --- END OF AJAX API HANDLER ---


// Check if user is logged in
if (!isLoggedIn()) {
    redirect('login.php');
}

// Initialize variables
$user = [];
$currentUserId = getCurrentUserId();

try {
    $pdo = getDBConnection();
    
    // Get user data and account data
    $stmt = $pdo->prepare("SELECT u.*, a.account_number, a.balance, a.account_type AS financial_account_type 
                          FROM users u 
                          LEFT JOIN accounts a ON u.id = a.user_id 
                          WHERE u.id = ? LIMIT 1");
    $stmt->execute([$currentUserId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    // If user record doesn't exist, set defaults
    if (!$user) {
         $user = ['id' => $currentUserId, 'full_name' => 'User', 'email' => 'your-email@example.com', 'account_type' => 'personal'];
    }
    
    // Check if user has premium card
    $stmt = $pdo->prepare("SELECT * FROM premium_cards WHERE user_id = ? AND status = 'active'");
    $stmt->execute([$currentUserId]);
    $premiumCard = $stmt->fetch(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Deposit error: " . $e->getMessage());
    // Set sane defaults to prevent HTML errors
    $user = [
        'full_name' => 'User',
        'email' => 'your-email@example.com',
        'balance' => 0,
        'account_number' => 'Error',
        'account_type' => 'personal',
        'financial_account_type' => 'Error'
    ];
    $premiumCard = null;
}

// Get user's first name
$firstName = 'User';
if (!empty($user['full_name']) && $user['full_name'] != 'User') {
    $nameParts = explode(' ', $user['full_name']);
    $firstName = $nameParts[0];
}

// Process deposit form submission
$depositSuccess = false;
$depositError = '';

// This logic now runs ONLY after the OTP is confirmed and the form is truly submitted
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['otp_code'])) {
    
    $submittedOtp = $_POST['otp_code'];
    
    // 1. Check if OTP is valid
    if (!isset($_SESSION['deposit_otp'])) {
        $depositError = "Your security token has expired. Please try again.";
    } elseif ($_SESSION['deposit_otp_expiry'] < time()) {
        $depositError = "Your OTP has expired. Please try again.";
    } elseif ($submittedOtp != $_SESSION['deposit_otp']) {
        $depositError = "Invalid OTP. Please try again.";
    } else {
        // --- OTP IS VALID ---
        
        $depositType = $_POST['deposit_type'] ?? '';
        $amount = $_POST['amount'] ?? '';
        $referenceNumber = $_POST['reference_number'] ?? '';
        $bankName = $_POST['bank_name'] ?? '';
        $checkNumber = $_POST['check_number'] ?? '';
        $description = $_POST['description'] ?? '';
        
        // 2. Re-validate form data (security best practice)
        if (empty($depositType) || empty($amount)) {
            $depositError = "Deposit type and amount are required.";
        } elseif (!is_numeric($amount) || $amount <= 0) {
            $depositError = "Please enter a valid amount.";
        } else {
            
            // --- DATABASE LOGIC FOR DEPOSIT ---
            $pdo->beginTransaction();
            try {
                // 1. Credit the user's account
                $stmt = $pdo->prepare("UPDATE accounts SET balance = balance + ? WHERE user_id = ?");
                $stmt->execute([$amount, $currentUserId]);

                // 2. Log the transaction in the history
                $transactionDescription = "Deposit via " . ucfirst($depositType);
                if (!empty($description)) {
                    $transactionDescription .= ": " . $description;
                }
                
                if ($depositType === 'bank_transfer' && !empty($referenceNumber)) {
                    $transactionDescription .= " (Reference: " . $referenceNumber . ")";
                } elseif ($depositType === 'check' && !empty($checkNumber)) {
                    $transactionDescription .= " (Check #: " . $checkNumber . ")";
                }
                
                $stmt = $pdo->prepare("INSERT INTO transactions (user_id, type, amount, description, status) 
                                       VALUES (?, 'credit', ?, ?, 'pending')");
                $stmt->execute([$currentUserId, $amount, $transactionDescription]);

                // 3. Commit the changes
                $pdo->commit();
                $depositSuccess = true;
                
                // Refresh user balance in the $user variable so the page updates
                $user['balance'] = $user['balance'] + $amount;

            } catch (Exception $e) {
                // 4. If anything fails, roll back
                $pdo->rollBack();
                error_log("Deposit Failed: " . $e->getMessage());
                $depositError = "Your deposit could not be processed due to a server error. Please try again.";
            }
            // --- END OF DATABASE LOGIC ---
        }
    }
    
    // Clear the OTP from session after a single attempt
    unset($_SESSION['deposit_otp']);
    unset($_SESSION['deposit_otp_expiry']);
    
} elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // This handles cases where the form is submitted without the OTP (e.g., if JS fails)
    $depositError = "An unexpected error occurred. Please enable JavaScript and try again.";
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Deposit - Airwaves Credit Union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        /* Custom scrollbar */
        ::-webkit-scrollbar {
            width: 6px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb {
            background: #d1d5db;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb:hover {
            background: #9ca3af;
        }
        
        /* Card hover effects */
        .card-hover {
            transition: all 0.3s ease;
        }
        
        .card-hover:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        }
        
        /* Gradient cards */
        .gradient-card {
            background: linear-gradient(135deg, #ffffff 0%, #f0fdf4 50%, #ffffff 100%);
            border: 1px solid #dcfce7;
        }
        
        /* Sidebar styling */
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        /* Premium card styling */
        .premium-card {
            background: linear-gradient(135deg, #09010f 0%, #facc15 100%);
        }
        
        /* Text shadow for better readability */
        .text-sharp {
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
        }
        
        /* Animations */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Form styling */
        .form-input {
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
        
        /* OTP Input */
        #otp-input {
            letter-spacing: 0.5em; /* Spreads out the numbers */
            font-weight: 600;
        }
        
        /* Loading spinner */
        .spinner {
            animation: spin 1s linear infinite;
        }
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        
        /* Deposit type selection */
        .deposit-type-option {
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .deposit-type-option.selected {
            border-color: #22c55e;
            background-color: #f0fdf4;
        }
        
        /* Conditional fields */
        .conditional-field {
            display: none;
        }
        
        .conditional-field.active {
            display: block;
            animation: slideDown 0.3s ease-out;
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Hamburger Menu (Mobile) -->
                    <button id="sidebar-toggle" class="md:hidden mr-3 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                    </div>
                </div>
                
                <!-- User Info & Logout (Right Side) -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($user['full_name'] ?? 'U', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($user['full_name'] ?? 'User'); ?></span>
                    </div>
                    <a href="logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Left Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen pt-16 transition-transform -translate-x-full bg-white border-r border-gray-200 md:translate-x-0 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-home text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="wire_transfer.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-paper-plane text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Wire Transfer</span>
                    </a>
                </li>
                <li>
                    <a href="deposit.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-wallet text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Deposit</span>
                    </a>
                </li>
                <li>
                    <a href="transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-receipt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">History</span>
                    </a>
                </li>
                <li>
                    <a href="track_money.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-line text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Track Money</span>
                    </a>
                </li>
                <li>
                    <a href="add_recipient.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-user-plus text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Add Recipient</span>
                    </a>
                </li>
                 <li>
                    <a href="premiumcard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-gem text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Premium Card</span>
                    </a>
                </li>
                <li>
                    <a href="support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer with decorative element -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="gradient-card rounded-xl p-4 text-center">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Secure Banking</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main id="main-content" class="md:ml-64 pt-16 h-full pb-24 md:pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
        
            <!-- Welcome Section - Clear and bold -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-3xl font-bold text-gray-900 text-sharp">Deposit Funds</h1>
                <p class="text-gray-600 mt-1 font-medium">Add money to your account via bank transfer or check.</p>
            </div>

            <!-- Main Content Layout -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                
                <!-- Deposit Form (2/3) -->
                <div class="lg:col-span-2">
                    <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            New Deposit
                        </h2>
                        
                        <!-- This div is for client-side JS validation errors -->
                        <div id="js-error-message" class="mb-6"></div>
                        
                        <?php if ($depositSuccess): ?>
                            <!-- Success Message -->
                            <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center">
                                <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center mr-4">
                                    <i class="fas fa-check text-green-600"></i>
                                </div>
                                <div>
                                    <p class="font-semibold text-green-800">Deposit Submitted Successfully!</p>
                                    <p class="text-sm text-green-700">Your deposit has been submitted and will be processed within 1-2 business days.</p>
                                </div>
                            </div>
                        <?php elseif ($depositError): ?>
                            <!-- Server-side Error Message (from POST reload) -->
                            <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center">
                                <div class="w-10 h-10 rounded-full bg-red-100 flex items-center justify-center mr-4">
                                    <i class="fas fa-exclamation-triangle text-red-600"></i>
                                </div>
                                <div>
                                    <p class="font-semibold text-red-800">Deposit Failed</p>
                                    <p class="text-sm text-red-700"><?php echo htmlspecialchars($depositError); ?></p>
                                </div>
                            </div>
                        <?php endif; ?>
                        
                        <!-- The Form -->
                        <form method="POST" action="deposit.php" id="deposit-form">
                            <div class="space-y-5">
                                <!-- Deposit Type Selection -->
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Deposit Method</label>
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div class="deposit-type-option border-2 border-gray-200 rounded-xl p-4 text-center" data-type="bank_transfer">
                                            <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-blue-100 flex items-center justify-center">
                                                <i class="fas fa-university text-blue-600 text-xl"></i>
                                            </div>
                                            <p class="font-semibold text-gray-900">Bank Transfer</p>
                                            <p class="text-xs text-gray-500 mt-1">Transfer from another bank</p>
                                        </div>
                                        <div class="deposit-type-option border-2 border-gray-200 rounded-xl p-4 text-center" data-type="check">
                                            <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-purple-100 flex items-center justify-center">
                                                <i class="fas fa-money-check text-purple-600 text-xl"></i>
                                            </div>
                                            <p class="font-semibold text-gray-900">Check</p>
                                            <p class="text-xs text-gray-500 mt-1">Deposit a physical check</p>
                                        </div>
                                    </div>
                                    <input type="hidden" name="deposit_type" id="deposit_type" value="">
                                </div>
                                
                                <!-- Bank Transfer Fields (Conditional) -->
                                <div id="bank-transfer-fields" class="conditional-field space-y-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Bank Name</label>
                                        <input type="text" name="bank_name" placeholder="Enter bank name" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none">
                                    </div>
                                    
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Reference Number</label>
                                        <input type="text" name="reference_number" placeholder="Enter transfer reference number" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none">
                                    </div>
                                </div>
                                
                                <!-- Check Deposit Fields (Conditional) -->
                                <div id="check-deposit-fields" class="conditional-field space-y-4">
                                    <div>
                                        <label class="block text-sm font-medium text-gray-700 mb-2">Check Number</label>
                                        <input type="text" name="check_number" placeholder="Enter check number" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none">
                                    </div>
                                </div>
                                
                                <!-- Amount -->
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Amount</label>
                                    <div class="relative">
                                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <span class="text-gray-500 font-medium">$</span>
                                        </div>
                                        <input type="number" name="amount" step="0.01" min="0.01" placeholder="0.00" class="form-input w-full rounded-xl border border-gray-300 pl-8 pr-4 py-3 focus:outline-none">
                                    </div>
                                </div>
                                
                                <!-- Description -->
                                <div>
                                    <label class="block text-sm font-medium text-gray-700 mb-2">Description (Optional)</label>
                                    <input type="text" name="description" placeholder="Add a note for this deposit" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none">
                                </div>
                                
                                <!-- Deposit Button -->
                                <div class="pt-4">
                                    <!-- This is now a 'button' type to trigger the modal, not 'submit' -->
                                    <button type="button" id="open-otp-modal-btn" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-3 px-4 rounded-xl transition-colors duration-300 flex items-center justify-center">
                                        <span id="btn-text">
                                            <i class="fas fa-plus-circle mr-2"></i>
                                            Submit Deposit
                                        </span>
                                        <span id="btn-loader" class="hidden">
                                            <i class="fas fa-spinner spinner mr-2"></i>
                                            Sending OTP...
                                        </span>
                                    </button>
                                </div>
                            </div>
                        </form>
                    </div>
                    
                    <!-- Deposit Information -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm mt-6 animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Deposit Information
                        </h2>
                        
                        <div class="space-y-4">
                            <div class="flex justify-between items-center p-4 rounded-xl bg-blue-50 border border-blue-100">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-lg bg-blue-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-university text-blue-600"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-semibold text-gray-900">Bank Transfer</p>
                                        <p class="text-xs text-gray-500 font-medium">Processing: 1-2 business days</p>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="flex justify-between items-center p-4 rounded-xl bg-purple-50 border border-purple-100">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-lg bg-purple-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-money-check text-purple-600"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-semibold text-gray-900">Check Deposit</p>
                                        <p class="text-xs text-gray-500 font-medium">Processing: 3-5 business days</p>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="flex justify-between items-center p-4 rounded-xl bg-green-50 border border-green-100">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mr-3">
                                        <i class="fas fa-shield-alt text-green-600"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-semibold text-gray-900">Security</p>
                                        <p class="text-xs text-gray-500 font-medium">256-bit encryption</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Sidebar Column (1/3) -->
                <div class="lg:col-span-1 space-y-6">
                    
                    <!-- Account Summary -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm border border-gray-100 animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Account Summary
                        </h2>
                        
                        <div class="space-y-4">
                            <div class="flex justify-between items-center">
                                <span class="text-gray-600 font-medium">Available Balance</span>
                                <span class="text-lg font-bold text-gray-900">$<?php echo number_format($user['balance'] ?? 0, 2); ?></span>
                            </div>
                            
                            <div class="flex justify-between items-center">
                                <span class="text-gray-600 font-medium">Account Number</span>
                                <span class="font-mono text-gray-900"><?php echo htmlspecialchars($user['account_number'] ?? 'N/A'); ?></span>
                            </div>
                            
                            <div class="flex justify-between items-center">
                                <span class="text-gray-600 font-medium">Account Type</span>
                                <span class="text-gray-900 font-medium">
                                    <?php 
                                        if ($premiumCard) {
                                            echo '<span class="text-green-600">Premium</span>';
                                        } else {
                                            echo htmlspecialchars(ucfirst($user['financial_account_type'] ?? 'Checking'));
                                        }
                                    ?>
                                </span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Recent Deposits -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm border border-gray-100 animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Recent Deposits
                        </h2>
                        
                        <div class="space-y-4">
                            <?php
                            try {
                                $stmt = $pdo->prepare("SELECT * FROM transactions WHERE user_id = ? AND type = 'credit' ORDER BY created_at DESC LIMIT 3");
                                $stmt->execute([$currentUserId]);
                                $recentDeposits = $stmt->fetchAll(PDO::FETCH_ASSOC);
                                
                                if (!empty($recentDeposits)): 
                                    foreach ($recentDeposits as $deposit): 
                            ?>
                                    <div class="flex items-center justify-between p-3 rounded-xl bg-white border border-gray-200">
                                        <div class="flex items-center">
                                            <div class="w-10 h-10 rounded-full bg-green-100 flex items-center justify-center mr-3">
                                                <i class="fas fa-wallet text-green-600"></i>
                                            </div>
                                            <div>
                                                <p class="text-sm font-semibold text-gray-900">$<?php echo number_format($deposit['amount'], 2); ?></p>
                                                <p class="text-xs text-gray-500 font-medium"><?php echo htmlspecialchars($deposit['description']); ?></p>
                                            </div>
                                        </div>
                                        <span class="text-xs font-medium px-2 py-1 rounded-full <?php echo $deposit['status'] === 'completed' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'; ?>">
                                            <?php echo ucfirst($deposit['status']); ?>
                                        </span>
                                    </div>
                            <?php 
                                    endforeach; 
                                else: 
                            ?>
                                <div class="text-center py-6">
                                    <div class="w-14 h-14 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                                        <i class="fas fa-wallet text-gray-400 text-xl"></i>
                                    </div>
                                    <p class="text-gray-500 font-medium">No recent deposits</p>
                                    <p class="text-sm text-gray-400 mt-1">Your deposit history will appear here</p>
                                </div>
                            <?php 
                                endif; 
                            } catch (PDOException $e) {
                                echo '<div class="text-center py-6 text-gray-500">Unable to load recent deposits</div>';
                            }
                            ?>
                        </div>
                    </div>
                    
                    <!-- Need Help? -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm border border-gray-100 animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-green-500 rounded-full mr-3"></span>
                            Need Help?
                        </h2>
                        
                        <div class="space-y-3">
                            <a href="support.php" class="flex items-center p-3 rounded-xl bg-white border border-gray-200 hover:border-green-500 transition-colors">
                                <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mr-3">
                                    <i class="fas fa-headset text-green-600"></i>
                                </div>
                                <div>
                                    <p class="text-sm font-semibold text-gray-900">Contact Support</p>
                                    <p class="text-xs text-gray-500 font-medium">24/7 customer service</p>
                                </div>
                            </a>
                            
                            <a href="#" class="flex items-center p-3 rounded-xl bg-white border border-gray-200 hover:border-green-500 transition-colors">
                                <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mr-3">
                                    <i class="fas fa-question-circle text-green-600"></i>
                                </div>
                                <div>
                                    <p class="text-sm font-semibold text-gray-900">FAQs</p>
                                    <p class="text-xs text-gray-500 font-medium">Common questions</p>
                                </div>
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- Mobile Navigation -->
    <div class="fixed bottom-0 left-0 right-0 md:hidden z-50">
        <div class="bg-white rounded-t-2xl shadow-lg border-t border-gray-200">
            <div class="flex justify-around items-center py-3">
                <a href="dashboard.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-home text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Dashboard</span>
                </a>
                <a href="wire_transfer.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-paper-plane text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Transfer</span>
                </a>
                <a href="deposit.php" class="flex flex-col items-center text-green-600">
                    <div class="w-10 h-10 rounded-lg bg-green-100 flex items-center justify-center mb-1">
                        <i class="fas fa-wallet text-lg"></i>
                    </div>
                    <span class="text-xs font-semibold">Deposit</span>
                </a>
                <a href="transactions.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-receipt text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">History</span>
                </a>
                <a href="profile.php" class="flex flex-col items-center text-gray-500 hover:text-green-600 transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                         <i class="fas fa-user-circle text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Profile</span>
                </a>
            </div>
        </div>
    </div>
    
    <!-- OTP Modal -->
    <div id="otp-modal" class="fixed inset-0 z-[60] flex items-center justify-center bg-black bg-opacity-60 hidden transition-opacity duration-300">
        <div class="bg-white rounded-xl shadow-2xl p-6 w-full max-w-md mx-4 animate-slide-down" style="animation-duration: 0.3s;">
            <div class="flex justify-between items-center mb-4">
                <h2 class="text-xl font-bold text-gray-900">Confirm Deposit</h2>
                <button id="close-otp-modal-btn" class="text-gray-400 hover:text-gray-600">
                    <i class="fas fa-times text-2xl"></i>
                </button>
            </div>
            <p class="text-gray-600 text-sm mb-4">
                For your security, a one-time password (OTP) has been sent to your registered email: 
                <strong class="text-gray-800"><?php echo htmlspecialchars($user['email'] ?? 'your-email@example.com'); ?></strong>.
            </p>
            <div class="space-y-4">
                <div>
                    <label for="otp-input" class="block text-sm font-medium text-gray-700 mb-2">Enter OTP <span id="otp-hint" class="text-gray-400"></span></label>
                    <input type="text" id="otp-input" name="otp" placeholder="------" class="form-input w-full rounded-xl border border-gray-300 px-4 py-3 focus:outline-none text-center tracking-widest" maxlength="6">
                    <p id="otp-error" class="text-red-600 text-sm mt-2"></p>
                </div>
                <button id="confirm-otp-btn" class="w-full bg-green-600 hover:bg-green-700 text-white font-semibold py-3 px-4 rounded-xl transition-colors duration-300 flex items-center justify-center">
                    <i class="fas fa-check-circle mr-2"></i>
                    Confirm & Deposit
                </button>
                <button type="button" id="resend-otp-btn" class="w-full text-center text-sm text-green-600 hover:underline">Resend OTP</button>
            </div>
        </div>
    </div>


    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebar-toggle');
            const sidebar = document.getElementById('sidebar');

            if (sidebarToggle && sidebar) {
                // ... (sidebar toggle logic as before) ...
                sidebarToggle.addEventListener('click', function(e) {
                    e.stopPropagation(); // Prevent click from bubbling up
                    sidebar.classList.toggle('-translate-x-full');
                    sidebar.classList.toggle('translate-x-0');
                });

                // Click outside to close
                document.addEventListener('click', function(e) {
                    // Check if the sidebar is open, and the click is not on the sidebar or the toggle button
                    if (sidebar.classList.contains('translate-x-0') && !sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        sidebar.classList.add('-translate-x-full');
                        sidebar.classList.remove('translate-x-0');
                    }
                });
            }
            
            // --- Deposit Type Selection Logic ---
            const depositTypeOptions = document.querySelectorAll('.deposit-type-option');
            const depositTypeInput = document.getElementById('deposit_type');
            const bankTransferFields = document.getElementById('bank-transfer-fields');
            const checkDepositFields = document.getElementById('check-deposit-fields');
            
            depositTypeOptions.forEach(option => {
                option.addEventListener('click', function() {
                    // Remove selected class from all options
                    depositTypeOptions.forEach(opt => {
                        opt.classList.remove('selected');
                        opt.classList.remove('border-green-500');
                        opt.classList.add('border-gray-200');
                    });
                    
                    // Add selected class to clicked option
                    this.classList.add('selected');
                    this.classList.remove('border-gray-200');
                    this.classList.add('border-green-500');
                    
                    // Set the deposit type value
                    const depositType = this.getAttribute('data-type');
                    depositTypeInput.value = depositType;
                    
                    // Show/hide conditional fields
                    if (depositType === 'bank_transfer') {
                        bankTransferFields.classList.add('active');
                        checkDepositFields.classList.remove('active');
                    } else if (depositType === 'check') {
                        checkDepositFields.classList.add('active');
                        bankTransferFields.classList.remove('active');
                    } else {
                        bankTransferFields.classList.remove('active');
                        checkDepositFields.classList.remove('active');
                    }
                });
            });
            
            // --- REAL OTP Modal & Form Logic ---
            
            // Get all modal elements
            const openOtpModalBtn = document.getElementById('open-otp-modal-btn');
            const otpModal = document.getElementById('otp-modal');
            const closeOtpModalBtn = document.getElementById('close-otp-modal-btn');
            const confirmOtpBtn = document.getElementById('confirm-otp-btn');
            const resendOtpBtn = document.getElementById('resend-otp-btn');
            const otpInput = document.getElementById('otp-input');
            const otpError = document.getElementById('otp-error');
            const otpHint = document.getElementById('otp-hint');
            
            // Get form elements
            const jsErrorDiv = document.getElementById('js-error-message');
            const depositForm = document.getElementById('deposit-form');
            
            // Get button states
            const btnText = document.getElementById('btn-text');
            const btnLoader = document.getElementById('btn-loader');

            // --- Helper function to show errors ---
            function showClientError(message) {
                jsErrorDiv.innerHTML = `
                    <div class="mb-6 p-4 bg-red-50 border border-red-200 rounded-xl flex items-center">
                        <div class="w-10 h-10 rounded-full bg-red-100 flex items-center justify-center mr-4">
                            <i class="fas fa-exclamation-triangle text-red-600"></i>
                        </div>
                        <div>
                            <p class="font-semibold text-red-800">Validation Error</p>
                            <p class="text-sm text-red-700">${message}</p>
                        </div>
                    </div>`;
            }

            // --- Helper function to set button loading state ---
            function setButtonLoading(isLoading) {
                if (isLoading) {
                    btnText.classList.add('hidden');
                    btnLoader.classList.remove('hidden');
                    openOtpModalBtn.disabled = true;
                } else {
                    btnText.classList.remove('hidden');
                    btnLoader.classList.add('hidden');
                    openOtpModalBtn.disabled = false;
                }
            }
            
            // --- 1. Open Modal / Send OTP Request ---
            async function handleSendOtp() {
                // 1. Get form values
                const depositType = depositTypeInput.value;
                const amountInput = document.querySelector('input[name="amount"]');
                const amount = parseFloat(amountInput.value) || 0;

                // 2. Clear previous errors
                jsErrorDiv.innerHTML = '';
                otpError.innerText = '';
                otpHint.innerText = '';
                
                // 3. Client-side Validation
                if (!depositType) {
                    showClientError("Please select a deposit method.");
                    return;
                }
                if (amount <= 0) {
                    showClientError("Please enter a valid amount.");
                    return;
                }
                
                // 4. Show loading state
                setButtonLoading(true);
                
                // 5. Send data to server to request OTP
                const formData = new FormData(depositForm);
                formData.append('action', 'send_deposit_otp');
                
                try {
                    const response = await fetch('deposit.php', {
                        method: 'POST',
                        body: formData
                    });
                    
                    if (!response.ok) {
                        throw new Error('Server error. Please try again.');
                    }
                    
                    const result = await response.json();
                    
                    if (result.status === 'success') {
                        // 6. Show modal if server request is successful
                        otpModal.classList.remove('hidden');
                        otpInput.focus();
                    } else {
                        // Show server-side validation error
                        showClientError(result.message || 'An unknown error occurred.');
                    }
                    
                } catch (error) {
                    showClientError(error.message);
                } finally {
                    // 7. Hide loading state
                    setButtonLoading(false);
                }
            }
            
            if (openOtpModalBtn) {
                openOtpModalBtn.addEventListener('click', handleSendOtp);
            }
            
            // --- 2. Resend OTP ---
            if (resendOtpBtn) {
                // We just re-run the same "send OTP" function
                resendOtpBtn.addEventListener('click', handleSendOtp);
            }

            // --- 3. Close Modal ---
            if (closeOtpModalBtn) {
                closeOtpModalBtn.addEventListener('click', () => {
                    otpModal.classList.add('hidden');
                });
            }

            // --- 4. Confirm OTP & Submit Form ---
            if (confirmOtpBtn) {
                confirmOtpBtn.addEventListener('click', function() {
                    const otp = otpInput.value;
                    
                    if (otp.length !== 6 || !/^\d+$/.test(otp)) {
                        otpError.innerText = 'Please enter a valid 6-digit OTP.';
                        return;
                    }
                    
                    // --- OTP is valid, submit the main form ---
                    otpError.innerText = '';
                    
                    // Create a new hidden input to add the OTP to the form
                    const otpField = document.createElement('input');
                    otpField.type = 'hidden';
                    otpField.name = 'otp_code';
                    otpField.value = otp;
                    
                    depositForm.appendChild(otpField);
                    
                    // Show final processing state
                    confirmOtpBtn.disabled = true;
                    confirmOtpBtn.innerHTML = '<i class="fas fa-spinner spinner mr-2"></i> Confirming...';
                    
                    // Submit the actual form
                    depositForm.submit();
                });
            }
        });
    </script>

</body>
</html>