<?php
session_start(); // This is required for isLoggedIn() to work
require_once 'config.php';

/**
 * Generates a unique account number.
 * Moved to the top for better code structure.
 *
 * @param PDO $pdo The PDO database connection object.
 * @return string The unique account number.
 */
function generateAccountNumber($pdo) {
    do {
        // Increased randomness from 4 to 5 digits
        $accountNumber = 'OX' . date('Ymd') . str_pad(rand(0, 99999), 5, '0', STR_PAD_LEFT);
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM accounts WHERE account_number = ?");
        $stmt->execute([$accountNumber]);
        $exists = $stmt->fetchColumn();
    } while ($exists > 0);
    
    return $accountNumber;
}

// Check if user is logged in
if (!isLoggedIn()) {
    redirect('login.php');
}

// Initialize variables to prevent errors
$user = [];
$transactions = [];
$premiumCard = null;
$currentUserId = getCurrentUserId(); // Get user ID once

try {
    $pdo = getDBConnection();
    
    // Get user data and account data with a LEFT JOIN
    // *** FIX: Aliased a.account_type to financial_account_type to avoid conflict with u.account_type ***
    $stmt = $pdo->prepare("SELECT u.*, a.account_number, a.balance, a.account_type AS financial_account_type 
                          FROM users u 
                          LEFT JOIN accounts a ON u.id = a.user_id 
                          WHERE u.id = ? LIMIT 1");
    $stmt->execute([$currentUserId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    // If user record doesn't exist (e.g., deleted but session active), set defaults
    if (!$user) {
         $user = ['id' => $currentUserId, 'full_name' => 'User', 'email' => '', 'account_type' => 'personal'];
    }

    // Auto-generate account number if it doesn't exist
    if (empty($user['account_number']) || $user['account_number'] == 'N/A') {
        $accountNumber = generateAccountNumber($pdo);
        
        // Check if an account record already exists
        $stmtCheck = $pdo->prepare("SELECT id FROM accounts WHERE user_id = ?");
        $stmtCheck->execute([$currentUserId]);
        $accountExists = $stmtCheck->fetch();

        // *** FIX: Determine new account type based on user's profile type ***
        $newAccountType = ($user['account_type'] == 'business') ? 'business' : 'checking';
        
        if ($accountExists) {
            // Account row exists, but number is missing. Update it.
            $stmt = $pdo->prepare("UPDATE accounts SET account_number = ?, account_type = ? WHERE user_id = ?");
            $stmt->execute([$accountNumber, $newAccountType, $currentUserId]);
        } else {
            // No account row exists. Create it.
            $stmt = $pdo->prepare("INSERT INTO accounts (user_id, account_number, balance, account_type) VALUES (?, ?, 0.00, ?)");
            $stmt->execute([$currentUserId, $accountNumber, $newAccountType]);
        }
        
        // Refresh user data to get the new account info
        // *** FIX: Added the same alias to the refresh query ***
        $stmt = $pdo->prepare("SELECT u.*, a.account_number, a.balance, a.account_type AS financial_account_type
                             FROM users u 
                             LEFT JOIN accounts a ON u.id = a.user_id 
                             WHERE u.id = ? LIMIT 1");
        $stmt->execute([$currentUserId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    // --- ADDED: Fetch recent transactions ---
    // Fetch recent transactions (Limit 5 for dashboard)
    $stmt = $pdo->prepare("SELECT * FROM transactions WHERE user_id = ? ORDER BY created_at DESC LIMIT 5");
    $stmt->execute([$currentUserId]);
    $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    // --- END: Fetch recent transactions ---
    
    // Check if user has premium card
    $stmt = $pdo->prepare("SELECT * FROM premium_cards WHERE user_id = ? AND status = 'active'");
    $stmt->execute([$currentUserId]);
    $premiumCard = $stmt->fetch(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    error_log("Dashboard error: " . $e->getMessage());
    // Set sane defaults to prevent HTML errors, but indicate a problem
    $user = [
        'full_name' => 'User',
        'email' => '',
        'balance' => 0,
        'account_number' => 'Error',
        'account_type' => 'personal',
        'financial_account_type' => 'Error'
    ];
    $transactions = [];
    $premiumCard = null;
}

// Get user's first name
$firstName = 'User';
if (!empty($user['full_name']) && $user['full_name'] != 'User') {
    $nameParts = explode(' ', $user['full_name']);
    $firstName = $nameParts[0];
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - AirWaves Credit Union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        /* Custom scrollbar */
        ::-webkit-scrollbar {
            width: 6px;
        }
        ::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb {
            background: #d1d5db;
            border-radius: 10px;
        }
        ::-webkit-scrollbar-thumb:hover {
            background: #9ca3af;
        }
        
        /* Card hover effects */
        .card-hover {
            transition: all 0.3s ease;
        }
        
        .card-hover:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
        }
        
        /* Gradient cards */
        .gradient-card {
            background: linear-gradient(135deg, #ffffff 0%, #f0f7ff 50%, #ffffff 100%);
            border: 1px solid #dbeafe;
        }
        
        /* Sidebar styling */
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(13, 59, 93, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(13, 59, 93, 0.15);
            border-left: 4px solid #0d3b5d;
        }
        
        /* Premium card styling */
        .premium-card {
            background: linear-gradient(135deg, #09010f 0%, #0d3b5d 100%);
        }
        
        /* Text shadow for better readability */
        .text-sharp {
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.05);
        }
        
        /* Animations */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Hamburger Menu (Mobile) -->
                    <button id="sidebar-toggle" class="md:hidden mr-3 text-gray-600 hover:text-blue-800 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-[#0d3b5d]">.</span>
                    </div>
                </div>
                
                <!-- User Info & Logout (Right Side) -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-[#0d3b5d] flex items-center justify-center text-white font-bold shadow-sm">
                            <?php echo strtoupper(substr($user['full_name'] ?? 'U', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($user['full_name'] ?? 'User'); ?></span>
                    </div>
                    <a href="logout.php" title="Logout" class="text-gray-600 hover:text-[#0d3b5d] transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Left Sidebar -->
    <aside id="sidebar" class="fixed top-0 left-0 z-40 w-64 h-screen pt-16 transition-transform -translate-x-full bg-white border-r border-gray-200 md:translate-x-0 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="dashboard.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-home text-[#0d3b5d] text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="wire_transfer.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-paper-plane text-gray-500 group-hover:text-[#0d3b5d] text-lg w-6"></i>
                        <span class="ml-3 font-medium">Wire Transfer</span>
                    </a>
                </li>
                <li>
                    <a href="deposit.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-wallet text-gray-500 group-hover:text-[#0d3b5d] text-lg w-6"></i>
                        <span class="ml-3 font-medium">Deposit</span>
                    </a>
                </li>
                <li>
                    <a href="transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-receipt text-gray-500 group-hover:text-[#0d3b5d] text-lg w-6"></i>
                        <span class="ml-3 font-medium">History</span>
                    </a>
                </li>
                <li>
                    <a href="track_money.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-line text-gray-500 group-hover:text-[#0d3b5d] text-lg w-6"></i>
                        <span class="ml-3 font-medium">Track Money</span>
                    </a>
                </li>
                <li>
                    <a href="add_recipient.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-user-plus text-gray-500 group-hover:text-[#0d3b5d] text-lg w-6"></i>
                        <span class="ml-3 font-medium">Add Recipient</span>
                    </a>
                </li>
                 <li>
                    <a href="premiumcard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-gem text-gray-500 group-hover:text-[#0d3b5d] text-lg w-6"></i>
                        <span class="ml-3 font-medium">Premium Card</span>
                    </a>
                </li>
                <li>
                    <a href="support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-[#0d3b5d] text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer with decorative element -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="gradient-card rounded-xl p-4 text-center">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-[#0d3b5d] flex items-center justify-center">
                        <i class="fas fa-shield-alt text-white"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Secure Banking</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main id="main-content" class="md:ml-64 pt-16 h-full pb-24 md:pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
        
            <!-- Welcome Section - Clear and bold -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-3xl font-bold text-gray-900 text-sharp">Hi <?php echo htmlspecialchars($firstName); ?>! 👋</h1>
                <p class="text-gray-600 mt-1 font-medium">Welcome back</p>
            </div>

            <!-- Modern Stats Cards Grid with Blue Gradient -->
            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-5 mb-8">
                
           
                <!-- Card 1: Total Balance with Currency Converter -->
<div class="gradient-card rounded-xl p-5 card-hover shadow-sm">
    <div class="flex justify-between items-start">
        <div>
            <p class="text-sm font-medium text-gray-600">Total Balance</p>
            <p class="text-2xl font-bold mt-1 text-gray-900 text-sharp" id="balanceDisplay">
                $<?php echo number_format($user['balance'] ?? 0, 2); ?>
            </p>
        </div>
        <div class="w-10 h-10 rounded-lg bg-[#0d3b5d] flex items-center justify-center">
            <i class="fas fa-wallet text-white text-lg"></i>
        </div>
    </div>
    
    <!-- Currency Converter -->
    <div class="mt-4">
        <label class="block text-xs font-medium text-gray-500 mb-2">View in different currency:</label>
        <div class="flex space-x-2">
            <select id="currencySelect" class="flex-1 text-xs border border-gray-300 rounded-lg px-2 py-1 focus:ring-1 focus:ring-[#0d3b5d] focus:border-[#0d3b5d]">
                <option value="USD" data-rate="1">USD ($)</option>
                <option value="EUR" data-rate="0.85">EUR (EUR)</option>
                <option value="GBP" data-rate="0.73">GBP (GBP)</option>
                <option value="AUD" data-rate="1.35">AUD (AUD)</option>
            </select>
            <button onclick="resetCurrency()" class="text-xs bg-gray-100 text-gray-600 px-2 py-1 rounded-lg hover:bg-gray-200 transition-colors">
                Reset
            </button>
        </div>
        <div id="conversionRate" class="text-xs text-gray-500 mt-2 hidden">
            Rate: 1 USD = <span id="rateValue">1</span> <span id="targetCurrency">USD</span>
        </div>
    </div>
    
    <div class="mt-3 flex items-center text-xs">
        <i class="fas fa-arrow-up text-[#0d3b5d] mr-1"></i>
        <span class="text-[#0d3b5d] font-medium">2.5% from last month</span>
    </div>
</div>
                <!-- Card 2: Account Number -->
                <div class="gradient-card rounded-xl p-5 card-hover shadow-sm">
                    <div class="flex justify-between items-start">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Account Number</p>
                            <p class="text-lg font-bold mt-1 text-gray-900 text-sharp font-mono"><?php echo htmlspecialchars($user['account_number'] ?? 'N/A'); ?></p>
                        </div>
                        <div class="w-10 h-10 rounded-lg bg-[#0d3b5d] flex items-center justify-center">
                            <i class="fas fa-hashtag text-white text-lg"></i>
                        </div>
                    </div>
                    <div class="mt-3 flex items-center text-xs">
                        <i class="fas fa-copy text-gray-500 mr-1"></i>
                        <span class="text-gray-500 font-medium">Click to copy</span>
                    </div>
                </div>
                
                <!-- Card 3: Account Type -->
                <div class="gradient-card rounded-xl p-5 card-hover shadow-sm">
                    <div class="flex justify-between items-start">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Account Type</p>
                            <p class="text-xl font-bold mt-1 text-gray-900 text-sharp">
                                <?php 
                                    if ($premiumCard) {
                                        echo '<span class="text-[#0d3b5d]">Premium</span>';
                                    } else {
                                        echo htmlspecialchars(ucfirst($user['financial_account_type'] ?? 'Checking'));
                                    }
                                ?>
                            </p>
                        </div>
                        <div class="w-10 h-10 rounded-lg bg-[#0d3b5d] flex items-center justify-center">
                            <i class="fas fa-user-tag text-white text-lg"></i>
                        </div>
                    </div>
                    <div class="mt-3 flex items-center text-xs">
                        <i class="fas fa-star text-[#0d3b5d] mr-1"></i>
                        <span class="text-[#0d3b5d] font-medium">Upgrade available</span>
                    </div>
                </div>

                <!-- Card 4: Premium Status -->
                 <div class="gradient-card rounded-xl p-5 card-hover shadow-sm">
                    <div class="flex justify-between items-start">
                        <div>
                            <p class="text-sm font-medium text-gray-600">Premium Status</p>
                            <?php if ($premiumCard): ?>
                                <p class="text-xl font-bold mt-1 text-[#0d3b5d] text-sharp">Active</p>
                            <?php else: ?>
                                <p class="text-xl font-bold mt-1 text-gray-500 text-sharp">Inactive</p>
                            <?php endif; ?>
                        </div>
                        <div class="w-10 h-10 rounded-lg bg-[#0d3b5d] flex items-center justify-center">
                            <i class="fas fa-gem text-white text-lg"></i>
                        </div>
                    </div>
                    <div class="mt-3 flex items-center text-xs">
                        <?php if ($premiumCard): ?>
                            <i class="fas fa-check-circle text-[#0d3b5d] mr-1"></i>
                            <span class="text-[#0d3b5d] font-medium">Active benefits</span>
                        <?php else: ?>
                            <i class="fas fa-lock text-gray-500 mr-1"></i>
                            <span class="text-gray-500 font-medium">Unlock premium</span>
                        <?php endif; ?>
                    </div>
                </div>

            </div>

            <!-- Main Dashboard Layout (2/3 + 1/3) -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">

                <!-- Main Column (2/3) -->
                <div class="lg:col-span-2 space-y-6">

                    <!-- Quick Actions -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-5 flex items-center">
                            <span class="w-1 h-6 bg-[#0d3b5d] rounded-full mr-3"></span>
                            Quick Actions
                        </h2>
                        <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
                            <a href="wire_transfer.php" class="flex flex-col items-center p-4 rounded-xl border border-gray-200 hover:border-[#0d3b5d] hover:bg-blue-50 transition-all duration-300 group">
                                <div class="w-12 h-12 bg-[#0d3b5d] rounded-xl flex items-center justify-center mb-3 group-hover:bg-[#0d3b5d] group-hover:scale-110 transition-all duration-300">
                                    <i class="fas fa-paper-plane text-white text-lg transition-colors"></i>
                                </div>
                                <span class="text-sm font-semibold text-gray-700 group-hover:text-[#0d3b5d] text-center transition-colors">Transfer</span>
                            </a>
                            <a href="deposit.php" class="flex flex-col items-center p-4 rounded-xl border border-gray-200 hover:border-[#0d3b5d] hover:bg-blue-50 transition-all duration-300 group">
                                <div class="w-12 h-12 bg-[#0d3b5d] rounded-xl flex items-center justify-center mb-3 group-hover:bg-[#0d3b5d] group-hover:scale-110 transition-all duration-300">
                                    <i class="fas fa-wallet text-white text-lg transition-colors"></i>
                                </div>
                                <span class="text-sm font-semibold text-gray-700 group-hover:text-[#0d3b5d] text-center transition-colors">Deposit</span>
                            </a>
                            <a href="premiumcard.php" class="flex flex-col items-center p-4 rounded-xl border border-gray-200 hover:border-[#0d3b5d] hover:bg-blue-50 transition-all duration-300 group">
                                <div class="w-12 h-12 bg-[#0d3b5d] rounded-xl flex items-center justify-center mb-3 group-hover:bg-[#0d3b5d] group-hover:scale-110 transition-all duration-300">
                                    <i class="fas fa-credit-card text-white text-lg transition-colors"></i>
                                </div>
                                <span class="text-sm font-semibold text-gray-700 group-hover:text-[#0d3b5d] text-center transition-colors">My Cards</span>
                            </a>
                            <a href="support.php" class="flex flex-col items-center p-4 rounded-xl border border-gray-200 hover:border-[#0d3b5d] hover:bg-blue-50 transition-all duration-300 group">
                                <div class="w-12 h-12 bg-[#0d3b5d] rounded-xl flex items-center justify-center mb-3 group-hover:bg-[#0d3b5d] group-hover:scale-110 transition-all duration-300">
                                    <i class="fas fa-headset text-white text-lg transition-colors"></i>
                                </div>
                                <span class="text-sm font-semibold text-gray-700 group-hover:text-[#0d3b5d] text-center transition-colors">Support</span>
                            </a>
                        </div>
                    </div>

                    <!-- Recent Transactions -->
                    <div class="gradient-card rounded-xl shadow-sm border border-gray-100 overflow-hidden animate-slide-up">
                        <div class="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
                            <h2 class="text-xl font-bold text-gray-900 flex items-center">
                                <span class="w-1 h-6 bg-[#0d3b5d] rounded-full mr-3"></span>
                                Recent Transactions
                            </h2>
                            <a href="transactions.php" class="text-sm text-[#0d3b5d] hover:text-blue-800 font-semibold flex items-center">
                                View all
                                <i class="fas fa-chevron-right ml-1 text-xs"></i>
                            </a>
                        </div>
                        <div class="divide-y divide-gray-100 transaction-list max-h-96 overflow-y-auto">
                            
                            <!-- === REPLACED PLACEHOLDER WITH DYNAMIC LIST === -->
                            <?php if (empty($transactions)): ?>
                                <!-- No Transactions Message -->
                                <div class="px-6 py-12 text-center">
                                    <div class="w-14 h-14 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                                        <i class="fas fa-file-invoice-dollar text-gray-400 text-xl"></i>
                                    </div>
                                    <p class="text-gray-500 font-medium">No transactions yet</p>
                                    <p class="text-sm text-gray-400 mt-1">Your recent activity will appear here</p>
                                </div>
                            <?php else: ?>
                                <!-- Transactions List -->
                                <div class="p-4 space-y-3">
                                    <?php foreach ($transactions as $tx): ?>
                                        <?php
                                            // Determine icon and amount color
                                            $isDebit = $tx['type'] == 'debit';
                                            $iconClass = $isDebit ? 'fa-arrow-up text-red-600' : 'fa-arrow-down text-green-600';
                                            $amountSign = $isDebit ? '-' : '+';
                                            $amountClass = $isDebit ? 'text-red-700' : 'text-green-700';

                                            // Format description
                                            $description = htmlspecialchars($tx['description']);
                                            if ($isDebit && !empty($tx['recipient_account'])) {
                                                $description = "To: " . htmlspecialchars($tx['recipient_account']);
                                            } elseif ($tx['type'] == 'deposit') {
                                                $description = "Deposit";
                                            } elseif (empty($description)) {
                                                $description = ucfirst($tx['type']);
                                            }
                                        ?>
                                        <!-- Single Transaction Item -->
                                        <div class="flex items-center justify-between p-3 rounded-xl bg-white border border-gray-200 shadow-sm">
                                            <div class="flex items-center min-w-0"> <!-- Added min-w-0 for truncate -->
                                                <div class="w-10 h-10 rounded-full <?php echo $isDebit ? 'bg-red-50' : 'bg-green-50'; ?> flex items-center justify-center mr-3 flex-shrink-0">
                                                    <i class="fas <?php echo $iconClass; ?>"></i>
                                                </div>
                                                <div class="min-w-0 truncate"> <!-- Added min-w-0 and truncate -->
                                                    <p class="text-sm font-semibold text-gray-900 truncate"><?php echo $description; ?></p>
                                                    <p class="text-xs text-gray-500 font-medium">
                                                        <?php echo date('M d, Y', strtotime($tx['created_at'])); ?>
                                                    </p>
                                                </div>
                                            </div>
                                            <div class="text-right flex-shrink-0 ml-2">
                                                <p class="text-sm font-bold <?php echo $amountClass; ?> transaction-amount" 
   data-original-amount="<?php echo $tx['amount']; ?>">
    <?php echo $amountSign; ?>$<?php echo number_format($tx['amount'], 2); ?>
</p>
                                                </p>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            <?php endif; ?>
                            <!-- === END OF REPLACEMENT === -->

                        </div>
                    </div>

                </div>

                <!-- Sidebar Column (1/3) -->
                <div class="lg:col-span-1 space-y-6">

                    <!-- Premium Card Section -->
                    <div class="gradient-card rounded-xl shadow-sm border border-gray-100 overflow-hidden animate-slide-up">
                        <div class="px-6 py-4 border-b border-gray-200 flex justify-between items-center">
                            <h2 class="text-xl font-bold text-gray-900 flex items-center">
                                <span class="w-1 h-6 bg-[#0d3b5d] rounded-full mr-3"></span>
                                Premium Card
                            </h2>
                            <?php if ($premiumCard): ?>
                                <span class="px-3 py-1 bg-[#0d3b5d] text-white text-xs font-semibold rounded-full">Active</span>
                            <?php else: ?>
                                <span class="px-3 py-1 bg-yellow-100 text-yellow-800 text-xs font-semibold rounded-full">Inactive</span>
                            <?php endif; ?>
                        </div>
                        <div class="p-6">
                            <?php if ($premiumCard): ?>
                                <!-- Active Card View -->
                                <div class="premium-card text-white rounded-xl p-5 mb-6 relative overflow-hidden">
                                    <div class="absolute top-0 right-0 w-20 h-20 bg-white/10 rounded-full -translate-y-1/2 translate-x-1/2"></div>
                                    <div class="absolute bottom-0 left-0 w-16 h-16 bg-white/10 rounded-full translate-y-1/2 -translate-x-1/2"></div>
                                    <div class="relative z-10">
                                        <div class="flex justify-between items-start mb-5">
                                            <span class="font-bold text-lg">Airwaves Premium</span>
                                            <i class="fas fa-gem text-lg"></i>
                                        </div>
                                        <div class="font-mono text-lg tracking-wider mb-5 font-semibold">
                                            **** **** **** <?php echo substr($premiumCard['card_number'], -4); ?>
                                        </div>
                                        <div class="flex justify-between text-sm">
                                            <div>
                                                <span class="text-xs opacity-80 block font-medium">Card Holder</span>
                                                <span class="font-semibold"><?php echo htmlspecialchars($user['full_name']); ?></span>
                                            </div>
                                            <div>
                                                <span class="text-xs opacity-80 block font-medium">Expires</span>
                                                <span class="font-semibold"><?php echo date('m/y', strtotime($premiumCard['expiry_date'])); ?></span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <a href="premiumcard.php" class="w-full flex justify-center items-center py-3 px-4 border border-gray-300 rounded-xl text-sm font-semibold text-gray-700 bg-white hover:bg-gray-50 hover:border-[#0d3b5d] transition-all duration-300 group">
                                    Manage Card
                                    <i class="fas fa-arrow-right ml-2 group-hover:translate-x-1 transition-transform"></i>
                                </a>
                            <?php else: ?>
                                <!-- Inactive Card View -->
                                <div class="text-center mb-6">
                                    <div class="w-16 h-16 bg-[#0d3b5d] rounded-xl flex items-center justify-center mx-auto mb-4">
                                        <i class="fas fa-credit-card text-white text-2xl"></i>
                                    </div>
                                    <h3 class="text-lg font-bold text-gray-900">Unlock Premium</h3>
                                    <p class="text-gray-600 mt-2 text-sm font-medium">Get exclusive benefits, lower fees, and a metal card.</p>
                                </div>
                                <a href="premiumcard.php" class="w-full flex justify-center items-center py-3 px-4 border border-transparent rounded-xl text-sm font-semibold text-white bg-[#0d3b5d] hover:bg-blue-900 transition-all duration-300 group">
                                    Activate Premium Card
                                    <i class="fas fa-arrow-right ml-2 group-hover:translate-x-1 transition-transform"></i>
                                </a>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Financial Insights -->
                    <div class="gradient-card rounded-xl p-6 shadow-sm border border-gray-100 animate-slide-up">
                        <h2 class="text-xl font-bold text-gray-900 mb-4 flex items-center">
                            <span class="w-1 h-6 bg-[#0d3b5d] rounded-full mr-3"></span>
                            Financial Insights
                        </h2>
                        <div class="space-y-4">
                            <div class="flex items-center justify-between p-3 rounded-xl bg-blue-50 border border-blue-100">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-lg bg-[#0d3b5d] flex items-center justify-center mr-3">
                                        <i class="fas fa-chart-line text-white"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-semibold text-gray-900">Spending Trend</p>
                                        <p class="text-xs text-gray-500 font-medium">Down 5% this month</p>
                                    </div>
                                </div>
                                <i class="fas fa-chevron-right text-gray-400"></i>
                            </div>
                            <div class="flex items-center justify-between p-3 rounded-xl bg-blue-50 border border-blue-100">
                                <div class="flex items-center">
                                    <div class="w-10 h-10 rounded-lg bg-[#0d3b5d] flex items-center justify-center mr-3">
                                        <i class="fas fa-piggy-bank text-white"></i>
                                    </div>
                                    <div>
                                        <p class="text-sm font-semibold text-gray-900">Savings Goal</p>
                                        <p class="text-xs text-gray-500 font-medium">75% completed</p>
                                    </div>
                                </div>
                                <i class="fas fa-chevron-right text-gray-400"></i>
                            </div>
                        </div>
                    </div>

                </div>
            </div>
        </div>
    </main>

    <!-- Mobile Navigation -->
    <div class="fixed bottom-0 left-0 right-0 md:hidden z-50">
        <div class="bg-white rounded-t-2xl shadow-lg border-t border-gray-200">
            <div class="flex justify-around items-center py-3">
                <a href="dashboard.php" class="flex flex-col items-center text-[#0d3b5d]">
                    <div class="w-10 h-10 rounded-lg bg-[#0d3b5d] flex items-center justify-center mb-1">
                        <i class="fas fa-home text-lg text-white"></i>
                    </div>
                    <span class="text-xs font-semibold">Dashboard</span>
                </a>
                <a href="wire_transfer.php" class="flex flex-col items-center text-gray-500 hover:text-[#0d3b5d] transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-paper-plane text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Transfer</span>
                </a>
                <a href="premiumcard.php" class="flex flex-col items-center text-gray-500 hover:text-[#0d3b5d] transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-credit-card text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Cards</span>
                </a>
                <a href="transactions.php" class="flex flex-col items-center text-gray-500 hover:text-[#0d3b5d] transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                        <i class="fas fa-receipt text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">History</span>
                </a>
                <a href="profile.php" class="flex flex-col items-center text-gray-500 hover:text-[#0d3b5d] transition-colors">
                    <div class="w-10 h-10 rounded-lg bg-gray-100 flex items-center justify-center mb-1">
                         <i class="fas fa-user-circle text-lg"></i>
                    </div>
                    <span class="text-xs font-medium">Profile</span>
                </a>
            </div>
        </div>
    </div>

    <script>
   // Currency conversion functionality
document.addEventListener('DOMContentLoaded', function() {
    const currencySelect = document.getElementById('currencySelect');
    const balanceDisplay = document.getElementById('balanceDisplay');
    const conversionRate = document.getElementById('conversionRate');
    const rateValue = document.getElementById('rateValue');
    const targetCurrency = document.getElementById('targetCurrency');
    
    // Get original balance from the displayed text
    const originalBalanceText = balanceDisplay.textContent;
    const originalBalance = parseFloat(originalBalanceText.replace(/[^0-9.-]+/g, ''));
    
    // Store original balance for reset
    balanceDisplay.setAttribute('data-original', originalBalanceText);
    
    currencySelect.addEventListener('change', function() {
        const selectedOption = this.options[this.selectedIndex];
        const rate = parseFloat(selectedOption.getAttribute('data-rate'));
        const currency = selectedOption.value;
        
        // Convert balance
        const convertedBalance = originalBalance * rate;
        
        // Update display with currency code instead of symbol
        balanceDisplay.textContent = `${currency} ${convertedBalance.toFixed(2)}`;
        
        // Show conversion rate
        rateValue.textContent = rate.toFixed(2);
        targetCurrency.textContent = currency;
        conversionRate.classList.remove('hidden');
        
        // Update all transaction amounts on the page
        updateTransactionCurrencies(rate, currency);
    });
});

function resetCurrency() {
    const balanceDisplay = document.getElementById('balanceDisplay');
    const originalBalance = balanceDisplay.getAttribute('data-original');
    const conversionRate = document.getElementById('conversionRate');
    const currencySelect = document.getElementById('currencySelect');
    
    // Reset balance display
    balanceDisplay.textContent = originalBalance;
    
    // Hide conversion rate
    conversionRate.classList.add('hidden');
    
    // Reset dropdown
    currencySelect.value = 'USD';
    
    // Reset transaction amounts
    resetTransactionCurrencies();
}

function updateTransactionCurrencies(rate, currency) {
    // Update all transaction amounts on the page
    document.querySelectorAll('.transaction-amount').forEach(element => {
        const originalAmount = element.getAttribute('data-original-amount');
        if (originalAmount) {
            const convertedAmount = parseFloat(originalAmount) * rate;
            element.textContent = `${currency} ${convertedAmount.toFixed(2)}`;
        }
    });
}

function resetTransactionCurrencies() {
    // Reset all transaction amounts to original
    document.querySelectorAll('.transaction-amount').forEach(element => {
        const originalAmount = element.getAttribute('data-original-amount');
        if (originalAmount) {
            element.textContent = `USD ${parseFloat(originalAmount).toFixed(2)}`;
        }
    });
}
        document.addEventListener('DOMContentLoaded', function() {
            const sidebarToggle = document.getElementById('sidebar-toggle');
            const sidebar = document.getElementById('sidebar');

            if (sidebarToggle && sidebar) {
                sidebarToggle.addEventListener('click', function(e) {
                    e.stopPropagation(); // Prevent click from bubbling up
                    sidebar.classList.toggle('-translate-x-full');
                    sidebar.classList.toggle('translate-x-0');
                });

                // Click outside to close
                document.addEventListener('click', function(e) {
                    // Check if the sidebar is open, and the click is not on the sidebar or the toggle button
                    if (sidebar.classList.contains('translate-x-0') && !sidebar.contains(e.target) && !sidebarToggle.contains(e.target)) {
                        sidebar.classList.add('-translate-x-full');
                        sidebar.classList.remove('translate-x-0');
                    }
                });
            }
        });
    </script>

</body>
</html>