<?php
// --- ADMIN AUTHENTICATION ---
session_start();
require_once 'config.php';

// Check if user is admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Only super admin can access settings
if ($_SESSION['admin_role'] !== 'super_admin') {
    header('Location: admin_dashboard.php');
    exit;
}

// Get admin user data
$admin_id = $_SESSION['admin_id'];
$pdo = getDBConnection();

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'update_settings') {
        try {
            $pdo->beginTransaction();
            
            foreach ($_POST['settings'] as $key => $value) {
                $stmt = $pdo->prepare("UPDATE system_settings SET setting_value = ?, updated_by = ? WHERE setting_key = ?");
                $stmt->execute([$value, $admin_id, $key]);
            }
            
            $pdo->commit();
            $_SESSION['admin_success'] = "Settings updated successfully!";
            
        } catch (Exception $e) {
            $pdo->rollBack();
            $_SESSION['admin_error'] = "Error updating settings: " . $e->getMessage();
        }
    }
    
    header('Location: admin_settings.php');
    exit;
}

// Get current settings
try {
    $stmt = $pdo->query("SELECT * FROM system_settings ORDER BY id");
    $settings = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get system info
    $stmt = $pdo->query("SELECT COUNT(*) as total_users FROM users");
    $total_users = $stmt->fetch(PDO::FETCH_ASSOC)['total_users'];
    
    $stmt = $pdo->query("SELECT COUNT(*) as total_transactions FROM transactions");
    $total_transactions = $stmt->fetch(PDO::FETCH_ASSOC)['total_transactions'];
    
    $stmt = $pdo->query("SELECT SUM(balance) as total_balance FROM accounts WHERE status = 'active'");
    $total_balance = $stmt->fetch(PDO::FETCH_ASSOC)['total_balance'] ?? 0;
    
} catch (PDOException $e) {
    error_log("Admin Settings error: " . $e->getMessage());
    $settings = [];
    $total_users = 0;
    $total_transactions = 0;
    $total_balance = 0;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>System Settings - Airwaves credit union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    <style>
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        .toggle-checkbox:checked {
            right: 0;
            border-color: #10b981;
        }
        
        .toggle-checkbox:checked + .toggle-label {
            background-color: #10b981;
        }
        
        /* Mobile responsive styles */
        .mobile-menu-btn {
            display: none;
        }
        
        @media (max-width: 1024px) {
            .mobile-menu-btn {
                display: block;
            }
            
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 40;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0 !important;
            }
            
            .overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 30;
            }
            
            .overlay.active {
                display: block;
            }
        }
        
        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: repeat(2, 1fr) !important;
            }
            
            .settings-grid {
                grid-template-columns: 1fr !important;
                gap: 1rem !important;
            }
            
            .card-padding {
                padding: 1rem !important;
            }
            
            .toggle-container {
                justify-content: space-between !important;
            }
        }
        
        @media (max-width: 640px) {
            .stats-grid {
                grid-template-columns: 1fr !important;
            }
            
            .nav-container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
            
            .main-container {
                padding: 1rem !important;
            }
        }
        
        /* Animation for mobile */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        /* Form styling improvements */
        .form-input {
            transition: all 0.3s ease;
        }
        
        .form-input:focus {
            border-color: #22c55e;
            box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.1);
        }
        
        /* Toggle switch improvements */
        .toggle-checkbox {
            transition: all 0.3s ease;
        }
        
        .toggle-label {
            transition: all 0.3s ease;
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden" style="font-family: 'Poppins', sans-serif;">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 nav-container">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Mobile menu button -->
                    <button class="mobile-menu-btn mr-4 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                        <span class="ml-2 text-sm font-medium text-gray-600 hidden sm:inline">Admin</span>
                    </div>
                </div>
                
                <!-- Admin Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($_SESSION['admin_username'] ?? 'A', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
                        <span class="text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full hidden sm:inline"><?php echo ucfirst($_SESSION['admin_role'] ?? 'admin'); ?></span>
                    </div>
                    <a href="admin_logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Mobile overlay -->
    <div class="overlay"></div>

    <!-- Left Sidebar -->
    <aside class="sidebar fixed top-0 left-0 z-40 w-64 h-screen pt-16 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="admin_dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-tachometer-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="admin_users.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-users text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">User Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-exchange-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Transactions</span>
                    </a>
                </li>
                <li>
                    <a href="admin_support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support Tickets</span>
                    </a>
                </li>
                <li>
                    <a href="admin_funds.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-money-bill-wave text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Fund Management</span>
                    </a>
                </li>
                <!-- NEW ACTIVATION CODES LINK -->
                <li>
                    <a href="admin_activation_codes.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-key text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Activation Codes</span>
                    </a>
                </li>
                <li>
                    <a href="admin_settings.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-cogs text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">System Settings</span>
                    </a>
                </li>
                <li>
                    <a href="admin_reports.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-chart-bar text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Reports & Analytics</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-4 text-center border border-green-100">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Super Admin Portal</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content ml-0 lg:ml-64 pt-16 h-full pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8 main-container">
        
            <!-- Page Header -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-2xl sm:text-3xl font-bold text-gray-900">System Settings ⚙️</h1>
                <p class="text-gray-600 mt-1 text-sm sm:text-base">Configure system-wide settings and preferences</p>
                <div class="mt-2 flex items-center text-xs sm:text-sm text-green-600">
                    <i class="fas fa-shield-alt mr-1"></i>
                    <span>Super Admin Access Only</span>
                </div>
            </div>

            <!-- Success/Error Messages -->
            <?php if (isset($_SESSION['admin_success'])): ?>
                <div class="mb-6 p-4 bg-green-50 border border-green-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-green-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-check text-green-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-green-800 text-sm sm:text-base">Success!</p>
                        <p class="text-xs sm:text-sm text-green-700"><?php echo htmlspecialchars($_SESSION['admin_success']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_success']); ?>
            <?php endif; ?>
            
            <?php if (isset($_SESSION['admin_error'])): ?>
                <div class="mb-6 p-4 bg-red-50 border-red-200 rounded-xl flex items-center animate-slide-down">
                    <div class="w-8 h-8 sm:w-10 sm:h-10 rounded-full bg-red-100 flex items-center justify-center mr-3 sm:mr-4">
                        <i class="fas fa-exclamation-triangle text-red-600 text-sm sm:text-base"></i>
                    </div>
                    <div>
                        <p class="font-semibold text-red-800 text-sm sm:text-base">Error</p>
                        <p class="text-xs sm:text-sm text-red-700"><?php echo htmlspecialchars($_SESSION['admin_error']); ?></p>
                    </div>
                </div>
                <?php unset($_SESSION['admin_error']); ?>
            <?php endif; ?>

            <!-- System Overview -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 sm:gap-6 mb-6 sm:mb-8 stats-grid">
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-blue-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Users</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $total_users; ?></p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-blue-100 flex items-center justify-center">
                            <i class="fas fa-users text-blue-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-green-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Balance</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1">$<?php echo number_format($total_balance, 2); ?></p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-green-100 flex items-center justify-center">
                            <i class="fas fa-wallet text-green-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-purple-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Transactions</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo $total_transactions; ?></p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-purple-100 flex items-center justify-center">
                            <i class="fas fa-exchange-alt text-purple-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Settings Form -->
            <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                <h2 class="text-lg sm:text-xl font-bold text-gray-900 mb-4 sm:mb-6 flex items-center">
                    <span class="w-1 h-4 sm:h-6 bg-green-500 rounded-full mr-2 sm:mr-3"></span>
                    System Configuration
                </h2>
                
                <?php if (empty($settings)): ?>
                    <div class="text-center py-8">
                        <div class="w-16 h-16 mx-auto mb-4 bg-gray-100 rounded-full flex items-center justify-center">
                            <i class="fas fa-cogs text-gray-400 text-xl"></i>
                        </div>
                        <p class="text-gray-500 font-medium">No system settings found</p>
                        <p class="text-sm text-gray-400 mt-1">System settings will appear here once configured</p>
                    </div>
                <?php else: ?>
                    <form method="POST" action="">
                        <input type="hidden" name="action" value="update_settings">
                        
                        <div class="space-y-4 sm:space-y-6">
                            <?php foreach ($settings as $setting): ?>
                                <div class="border-b border-gray-200 pb-4 sm:pb-6 last:border-b-0">
                                    <div class="grid grid-cols-1 md:grid-cols-3 gap-3 sm:gap-4 settings-grid">
                                        <div class="md:col-span-1">
                                            <label class="block text-sm font-medium text-gray-700 mb-1 sm:mb-2">
                                                <?php 
                                                    $label = str_replace('_', ' ', $setting['setting_key']);
                                                    echo ucwords($label);
                                                ?>
                                            </label>
                                            <?php if ($setting['description']): ?>
                                                <p class="text-xs sm:text-sm text-gray-500"><?php echo htmlspecialchars($setting['description']); ?></p>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <div class="md:col-span-2">
                                            <?php if ($setting['setting_type'] === 'boolean'): ?>
                                                <!-- Toggle Switch for Boolean -->
                                                <div class="flex items-center toggle-container">
                                                    <div class="relative inline-block w-12 mr-2 align-middle select-none">
                                                        <input type="checkbox" name="settings[<?php echo $setting['setting_key']; ?>]" 
                                                               value="1" id="toggle-<?php echo $setting['id']; ?>"
                                                               class="toggle-checkbox absolute block w-6 h-6 rounded-full bg-white border-4 appearance-none cursor-pointer"
                                                               <?php echo $setting['setting_value'] == '1' ? 'checked' : ''; ?>>
                                                        <label for="toggle-<?php echo $setting['id']; ?>" 
                                                               class="toggle-label block overflow-hidden h-6 rounded-full bg-gray-300 cursor-pointer"></label>
                                                    </div>
                                                    <label for="toggle-<?php echo $setting['id']; ?>" class="text-sm text-gray-700">
                                                        <?php echo $setting['setting_value'] == '1' ? 'Enabled' : 'Disabled'; ?>
                                                    </label>
                                                </div>
                                            <?php elseif ($setting['setting_type'] === 'number'): ?>
                                                <!-- Number Input -->
                                                <input type="number" name="settings[<?php echo $setting['setting_key']; ?>]" 
                                                       value="<?php echo htmlspecialchars($setting['setting_value']); ?>"
                                                       class="form-input w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm sm:text-base">
                                            <?php elseif ($setting['setting_type'] === 'textarea'): ?>
                                                <!-- Textarea Input -->
                                                <textarea name="settings[<?php echo $setting['setting_key']; ?>]" 
                                                          rows="3"
                                                          class="form-input w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm sm:text-base"><?php echo htmlspecialchars($setting['setting_value']); ?></textarea>
                                            <?php else: ?>
                                                <!-- Text Input -->
                                                <input type="text" name="settings[<?php echo $setting['setting_key']; ?>]" 
                                                       value="<?php echo htmlspecialchars($setting['setting_value']); ?>"
                                                       class="form-input w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm sm:text-base">
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                            
                            <!-- Submit Button -->
                            <div class="flex justify-end pt-4 sm:pt-6">
                                <button type="submit" 
                                        class="bg-green-600 text-white px-6 sm:px-8 py-2 sm:py-3 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm sm:text-base shadow-sm w-full sm:w-auto">
                                    <i class="fas fa-save mr-2"></i>
                                    Save Settings
                                </button>
                            </div>
                        </div>
                    </form>
                <?php endif; ?>
            </div>
            
            <!-- Quick Actions -->
            <div class="mt-6 bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl p-4 sm:p-6 border border-blue-100 animate-slide-up">
                <h3 class="text-lg font-semibold text-gray-900 mb-4 flex items-center">
                    <i class="fas fa-bolt text-blue-600 mr-2"></i>
                    Quick Actions
                </h3>
                <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 text-sm">
                    <a href="admin_backup.php" class="flex items-center p-3 bg-white rounded-lg border border-blue-200 hover:border-blue-500 transition-colors">
                        <div class="w-8 h-8 rounded-full bg-blue-100 flex items-center justify-center mr-3">
                            <i class="fas fa-database text-blue-600"></i>
                        </div>
                        <div>
                            <p class="font-medium text-gray-900">Backup</p>
                            <p class="text-xs text-gray-500">System backup</p>
                        </div>
                    </a>
                    <a href="admin_logs.php" class="flex items-center p-3 bg-white rounded-lg border border-green-200 hover:border-green-500 transition-colors">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center mr-3">
                            <i class="fas fa-file-alt text-green-600"></i>
                        </div>
                        <div>
                            <p class="font-medium text-gray-900">Logs</p>
                            <p class="text-xs text-gray-500">System logs</p>
                        </div>
                    </a>
                    <a href="admin_maintenance.php" class="flex items-center p-3 bg-white rounded-lg border border-yellow-200 hover:border-yellow-500 transition-colors">
                        <div class="w-8 h-8 rounded-full bg-yellow-100 flex items-center justify-center mr-3">
                            <i class="fas fa-tools text-yellow-600"></i>
                        </div>
                        <div>
                            <p class="font-medium text-gray-900">Maintenance</p>
                            <p class="text-xs text-gray-500">System tools</p>
                        </div>
                    </a>
                    <a href="admin_security.php" class="flex items-center p-3 bg-white rounded-lg border border-red-200 hover:border-red-500 transition-colors">
                        <div class="w-8 h-8 rounded-full bg-red-100 flex items-center justify-center mr-3">
                            <i class="fas fa-shield-alt text-red-600"></i>
                        </div>
                        <div>
                            <p class="font-medium text-gray-900">Security</p>
                            <p class="text-xs text-gray-500">Security settings</p>
                        </div>
                    </a>
                </div>
            </div>
        </div>
    </main>

    <script>
        // Mobile menu functionality
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.overlay');
            
            function toggleSidebar() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
                document.body.classList.toggle('overflow-hidden');
            }
            
            if (mobileMenuBtn) {
                mobileMenuBtn.addEventListener('click', toggleSidebar);
            }
            if (overlay) {
                overlay.addEventListener('click', toggleSidebar);
            }
            
            // Handle window resize
            function handleResize() {
                if (window.innerWidth >= 1024) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                    document.body.classList.remove('overflow-hidden');
                }
            }
            
            window.addEventListener('resize', handleResize);
            
            // Update toggle labels when toggled
            document.querySelectorAll('.toggle-checkbox').forEach(checkbox => {
                checkbox.addEventListener('change', function() {
                    const label = this.nextElementSibling.nextElementSibling;
                    label.textContent = this.checked ? 'Enabled' : 'Disabled';
                });
            });
            
            // Auto-hide success/error messages after 5 seconds
            setTimeout(() => {
                const messages = document.querySelectorAll('[class*="bg-green-50"], [class*="bg-red-50"]');
                messages.forEach(message => {
                    message.style.opacity = '0';
                    message.style.transition = 'opacity 0.5s ease';
                    setTimeout(() => message.remove(), 500);
                });
            }, 5000);
        });
    </script>

</body>
</html>