<?php
// --- ADMIN AUTHENTICATION ---
session_start();
require_once 'config.php';

// Check if user is admin
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

// Get admin user data
$admin_id = $_SESSION['admin_id'];
$pdo = getDBConnection();

// Date range filtering
$start_date = $_GET['start_date'] ?? date('Y-m-01'); // First day of current month
$end_date = $_GET['end_date'] ?? date('Y-m-t'); // Last day of current month
$report_type = $_GET['report_type'] ?? 'overview';

try {
    // Overall statistics
    $stmt = $pdo->query("SELECT COUNT(*) as total_users FROM users");
    $total_users = $stmt->fetch(PDO::FETCH_ASSOC)['total_users'];
    
    $stmt = $pdo->query("SELECT COUNT(*) as active_users FROM users WHERE is_active = 1");
    $active_users = $stmt->fetch(PDO::FETCH_ASSOC)['active_users'];
    
    $stmt = $pdo->query("SELECT COUNT(*) as total_transactions FROM transactions");
    $total_transactions = $stmt->fetch(PDO::FETCH_ASSOC)['total_transactions'];
    
    $stmt = $pdo->query("SELECT SUM(balance) as total_balance FROM accounts WHERE status = 'active'");
    $total_balance = $stmt->fetch(PDO::FETCH_ASSOC)['total_balance'] ?? 0;
    
    // Date-filtered statistics
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as transactions_count,
            SUM(CASE WHEN type = 'credit' THEN amount ELSE 0 END) as total_credits,
            SUM(CASE WHEN type = 'debit' THEN amount ELSE 0 END) as total_debits,
            SUM(CASE WHEN type = 'credit' THEN amount ELSE -amount END) as net_flow
        FROM transactions 
        WHERE DATE(created_at) BETWEEN ? AND ?
    ");
    $stmt->execute([$start_date, $end_date]);
    $period_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Transaction trends (last 30 days)
    $stmt = $pdo->prepare("
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as transaction_count,
            SUM(CASE WHEN type = 'credit' THEN amount ELSE 0 END) as credits,
            SUM(CASE WHEN type = 'debit' THEN amount ELSE 0 END) as debits
        FROM transactions 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        ORDER BY date
    ");
    $stmt->execute();
    $daily_trends = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // User registration trends
    $stmt = $pdo->prepare("
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as registrations
        FROM users 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        ORDER BY date
    ");
    $stmt->execute();
    $registration_trends = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Top users by balance
    $stmt = $pdo->prepare("
        SELECT u.full_name, u.email, a.account_number, a.balance, a.currency
        FROM users u 
        JOIN accounts a ON u.id = a.user_id 
        WHERE u.is_active = 1 AND a.status = 'active'
        ORDER BY a.balance DESC 
        LIMIT 10
    ");
    $stmt->execute();
    $top_users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Transaction types breakdown
    $stmt = $pdo->prepare("
        SELECT 
            type,
            COUNT(*) as count,
            SUM(amount) as total_amount
        FROM transactions 
        WHERE DATE(created_at) BETWEEN ? AND ?
        GROUP BY type
    ");
    $stmt->execute([$start_date, $end_date]);
    $transaction_types = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Currency distribution
    $stmt = $pdo->query("
        SELECT 
            currency,
            COUNT(*) as account_count,
            SUM(balance) as total_balance
        FROM accounts 
        WHERE status = 'active'
        GROUP BY currency
    ");
    $currency_distribution = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Support ticket statistics
    $stmt = $pdo->query("
        SELECT 
            status,
            COUNT(*) as count
        FROM support_tickets 
        GROUP BY status
    ");
    $ticket_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    error_log("Admin Reports error: " . $e->getMessage());
    $total_users = $active_users = $total_transactions = $total_balance = 0;
    $period_stats = ['transactions_count' => 0, 'total_credits' => 0, 'total_debits' => 0, 'net_flow' => 0];
    $daily_trends = $registration_trends = $top_users = $transaction_types = $currency_distribution = $ticket_stats = [];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Reports & Analytics - Airwaves credit union</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <style>
        .sidebar-item {
            transition: all 0.3s ease;
            border-radius: 12px;
            margin-bottom: 6px;
        }
        
        .sidebar-item:hover {
            background: rgba(34, 197, 94, 0.1);
            transform: translateX(5px);
        }
        
        .sidebar-item.active {
            background: rgba(34, 197, 94, 0.15);
            border-left: 4px solid #22c55e;
        }
        
        .stat-card {
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px -5px rgba(0, 0, 0, 0.1);
        }
        
        .chart-container {
            position: relative;
            height: 300px;
            width: 100%;
        }

        /* Mobile responsive styles */
        .mobile-menu-btn {
            display: none;
        }
        
        @media (max-width: 1024px) {
            .mobile-menu-btn {
                display: block;
            }
            
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
                z-index: 40;
            }
            
            .sidebar.active {
                transform: translateX(0);
            }
            
            .main-content {
                margin-left: 0 !important;
            }
            
            .overlay {
                display: none;
                position: fixed;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background-color: rgba(0, 0, 0, 0.5);
                z-index: 30;
            }
            
            .overlay.active {
                display: block;
            }
        }
        
        @media (max-width: 768px) {
            .stats-grid {
                grid-template-columns: repeat(2, 1fr) !important;
            }
            
            .charts-grid {
                grid-template-columns: 1fr !important;
            }
            
            .analytics-grid {
                grid-template-columns: 1fr !important;
            }
            
            .tables-grid {
                grid-template-columns: 1fr !important;
            }
            
            .table-responsive {
                overflow-x: auto;
                -webkit-overflow-scrolling: touch;
            }
            
            .table-responsive table {
                min-width: 600px;
            }
            
            .filter-grid {
                grid-template-columns: 1fr !important;
            }
            
            .card-padding {
                padding: 1rem !important;
            }
        }
        
        @media (max-width: 640px) {
            .stats-grid {
                grid-template-columns: 1fr !important;
            }
            
            .nav-container {
                padding-left: 1rem !important;
                padding-right: 1rem !important;
            }
            
            .main-container {
                padding: 1rem !important;
            }
            
            .chart-container {
                height: 250px;
            }
        }

        /* Animation for mobile */
        .animate-slide-down {
            animation: slideDown 0.5s ease-out;
        }
        
        .animate-slide-up {
            animation: slideUp 0.5s ease-out;
        }
        
        @keyframes slideDown {
            from {
                transform: translateY(-20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
        
        @keyframes slideUp {
            from {
                transform: translateY(20px);
                opacity: 0;
            }
            to {
                transform: translateY(0);
                opacity: 1;
            }
        }
    </style>
</head>
<body class="bg-gray-50 font-sans text-gray-900 min-h-screen overflow-x-hidden">
    
    <!-- Top Navigation -->
    <nav class="bg-white shadow-sm fixed top-0 left-0 right-0 z-50 border-b border-gray-200">
        <div class="max-w-full mx-auto px-4 sm:px-6 lg:px-8 nav-container">
            <div class="flex justify-between h-16">
                <div class="flex items-center">
                    <!-- Mobile menu button -->
                    <button class="mobile-menu-btn mr-4 text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-bars text-xl"></i>
                    </button>
                    
                    <!-- Logo -->
                    <div class="flex items-center">
                        <span class="text-2xl font-black text-gray-900">AIRWAVES</span>
                        <span class="text-2xl font-black text-green-600">.</span>
                        <span class="ml-2 text-sm font-medium text-gray-600 hidden sm:inline">Admin</span>
                    </div>
                </div>
                
                <!-- Admin Info & Logout -->
                <div class="flex items-center space-x-4">
                    <div class="flex items-center space-x-2">
                        <div class="w-8 h-8 rounded-full bg-green-100 flex items-center justify-center text-green-700 font-bold shadow-sm">
                            <?php echo strtoupper(substr($_SESSION['admin_username'] ?? 'A', 0, 1)); ?>
                        </div>
                        <span class="text-gray-700 hidden md:inline font-medium"><?php echo htmlspecialchars($_SESSION['admin_username'] ?? 'Admin'); ?></span>
                    </div>
                    <a href="admin_logout.php" title="Logout" class="text-gray-600 hover:text-green-600 transition-colors">
                        <i class="fas fa-sign-out-alt text-lg"></i>
                    </a>
                </div>
            </div>
        </div>
    </nav>

    <!-- Mobile overlay -->
    <div class="overlay"></div>

    <!-- Left Sidebar -->
    <aside class="sidebar fixed top-0 left-0 z-40 w-64 h-screen pt-16 bg-white border-r border-gray-200 overflow-y-auto">
        <div class="h-full px-4 py-6">
            <ul class="space-y-1 font-medium">
                <li>
                    <a href="admin_dashboard.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-tachometer-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Dashboard</span>
                    </a>
                </li>
                <li>
                    <a href="admin_users.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-users text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">User Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_transactions.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-exchange-alt text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Transactions</span>
                    </a>
                </li>
                <li>
                    <a href="admin_support.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-headset text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Support Tickets</span>
                    </a>
                </li>
                <li>
                    <a href="admin_funds.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-money-bill-wave text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">Fund Management</span>
                    </a>
                </li>
                <li>
                    <a href="admin_settings.php" class="sidebar-item flex items-center p-3 text-gray-600 rounded-lg hover:text-gray-900 group">
                        <i class="fas fa-cogs text-gray-500 group-hover:text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-medium">System Settings</span>
                    </a>
                </li>
                <li>
                    <a href="admin_reports.php" class="sidebar-item active flex items-center p-3 text-gray-900 rounded-lg group">
                        <i class="fas fa-chart-bar text-green-600 text-lg w-6"></i>
                        <span class="ml-3 font-semibold">Reports & Analytics</span>
                    </a>
                </li>
            </ul>
            
            <!-- Sidebar footer -->
            <div class="absolute bottom-6 left-0 right-0 px-4">
                <div class="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-4 text-center border border-green-100">
                    <div class="w-12 h-12 mx-auto mb-2 rounded-full bg-green-100 flex items-center justify-center">
                        <i class="fas fa-shield-alt text-green-600"></i>
                    </div>
                    <p class="text-xs text-gray-600 font-medium">Admin Portal</p>
                </div>
            </div>
        </div>
    </aside>

    <!-- Main Content -->
    <main class="main-content ml-0 lg:ml-64 pt-16 h-full pb-8">
        <div class="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8 main-container">
        
            <!-- Page Header -->
            <div class="mb-6 animate-slide-down">
                <h1 class="text-2xl sm:text-3xl font-bold text-gray-900">Reports & Analytics 📊</h1>
                <p class="text-gray-600 mt-1 text-sm sm:text-base">Comprehensive system analytics and performance reports</p>
            </div>

            <!-- Date Range Filter -->
            <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 mb-6 animate-slide-up card-padding">
                <h2 class="text-lg sm:text-xl font-bold text-gray-900 mb-4 flex items-center">
                    <span class="w-1 h-4 sm:h-6 bg-green-500 rounded-full mr-2 sm:mr-3"></span>
                    Report Filters
                </h2>
                <form method="GET" action="" class="grid grid-cols-1 md:grid-cols-4 gap-4 filter-grid">
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Report Type</label>
                        <select name="report_type" class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm sm:text-base">
                            <option value="overview" <?php echo $report_type === 'overview' ? 'selected' : ''; ?>>Overview</option>
                            <option value="financial" <?php echo $report_type === 'financial' ? 'selected' : ''; ?>>Financial</option>
                            <option value="users" <?php echo $report_type === 'users' ? 'selected' : ''; ?>>User Analytics</option>
                        </select>
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">Start Date</label>
                        <input type="date" name="start_date" value="<?php echo htmlspecialchars($start_date); ?>" 
                               class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm sm:text-base">
                    </div>
                    <div>
                        <label class="block text-sm font-medium text-gray-700 mb-2">End Date</label>
                        <input type="date" name="end_date" value="<?php echo htmlspecialchars($end_date); ?>" 
                               class="w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-green-500 focus:border-green-500 text-sm sm:text-base">
                    </div>
                    <div class="flex items-end">
                        <button type="submit" class="w-full bg-green-600 text-white px-4 sm:px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm sm:text-base">
                            <i class="fas fa-filter mr-2"></i>
                            Apply Filters
                        </button>
                    </div>
                </form>
            </div>

            <!-- Key Metrics -->
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6 mb-6 sm:mb-8 stats-grid">
                <div class="stat-card bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-blue-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Users</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo number_format($total_users); ?></p>
                            <p class="text-xs text-green-600 font-medium mt-1">
                                <i class="fas fa-users mr-1"></i>
                                <?php echo number_format($active_users); ?> active
                            </p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-blue-100 flex items-center justify-center">
                            <i class="fas fa-users text-blue-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="stat-card bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-green-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Total Balance</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1">$<?php echo number_format($total_balance, 2); ?></p>
                            <p class="text-xs text-gray-500 font-medium mt-1">Across all accounts</p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-green-100 flex items-center justify-center">
                            <i class="fas fa-wallet text-green-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="stat-card bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-purple-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Transactions</p>
                            <p class="text-xl sm:text-2xl font-bold text-gray-900 mt-1"><?php echo number_format($total_transactions); ?></p>
                            <p class="text-xs text-purple-600 font-medium mt-1">
                                <i class="fas fa-exchange-alt mr-1"></i>
                                <?php echo number_format($period_stats['transactions_count']); ?> in period
                            </p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-purple-100 flex items-center justify-center">
                            <i class="fas fa-exchange-alt text-purple-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="stat-card bg-white rounded-xl p-4 sm:p-6 shadow-sm border-l-4 border-orange-500 animate-slide-up">
                    <div class="flex items-center justify-between">
                        <div>
                            <p class="text-xs sm:text-sm font-medium text-gray-600">Net Flow</p>
                            <p class="text-xl sm:text-2xl font-bold <?php echo $period_stats['net_flow'] >= 0 ? 'text-green-600' : 'text-red-600'; ?> mt-1">
                                $<?php echo number_format($period_stats['net_flow'] ?? 0, 2); ?>
                            </p>
                            <p class="text-xs text-gray-500 font-medium mt-1">
                                <?php echo date('M j', strtotime($start_date)); ?> - <?php echo date('M j, Y', strtotime($end_date)); ?>
                            </p>
                        </div>
                        <div class="w-10 h-10 sm:w-12 sm:h-12 rounded-full bg-orange-100 flex items-center justify-center">
                            <i class="fas fa-chart-line text-orange-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Charts and Detailed Analytics -->
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 sm:gap-8 mb-6 sm:mb-8 charts-grid">
                <!-- Transaction Trends Chart -->
                <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                    <h3 class="text-base sm:text-lg font-bold text-gray-900 mb-4">Transaction Trends (Last 30 Days)</h3>
                    <div class="chart-container">
                        <canvas id="transactionTrendsChart"></canvas>
                    </div>
                </div>

                <!-- Transaction Types Chart -->
                <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                    <h3 class="text-base sm:text-lg font-bold text-gray-900 mb-4">Transaction Types</h3>
                    <div class="chart-container">
                        <canvas id="transactionTypesChart"></canvas>
                    </div>
                </div>
            </div>

            <!-- Additional Analytics -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 sm:gap-8 mb-6 sm:mb-8 analytics-grid">
                <!-- User Registration Trends -->
                <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                    <h3 class="text-base sm:text-lg font-bold text-gray-900 mb-4">User Registrations (Last 30 Days)</h3>
                    <div class="chart-container">
                        <canvas id="registrationChart"></canvas>
                    </div>
                </div>

                <!-- Currency Distribution -->
                <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                    <h3 class="text-base sm:text-lg font-bold text-gray-900 mb-4">Currency Distribution</h3>
                    <div class="chart-container">
                        <canvas id="currencyChart"></canvas>
                    </div>
                </div>

                <!-- Support Tickets -->
                <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                    <h3 class="text-base sm:text-lg font-bold text-gray-900 mb-4">Support Tickets Status</h3>
                    <div class="chart-container">
                        <canvas id="ticketsChart"></canvas>
                    </div>
                </div>
            </div>

            <!-- Top Users and Detailed Tables -->
            <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 sm:gap-8 tables-grid">
                <!-- Top Users by Balance -->
                <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                    <h3 class="text-base sm:text-lg font-bold text-gray-900 mb-4">Top Users by Balance</h3>
                    <div class="table-responsive">
                        <table class="min-w-full divide-y divide-gray-200">
                            <thead class="bg-gray-50">
                                <tr>
                                    <th class="px-3 sm:px-4 py-2 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">User</th>
                                    <th class="px-3 sm:px-4 py-2 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Account</th>
                                    <th class="px-3 sm:px-4 py-2 sm:py-3 text-left text-xs font-medium text-gray-500 uppercase">Balance</th>
                                </tr>
                            </thead>
                            <tbody class="bg-white divide-y divide-gray-200">
                                <?php if (!empty($top_users)): ?>
                                    <?php foreach ($top_users as $user): ?>
                                        <tr class="hover:bg-gray-50 transition-colors">
                                            <td class="px-3 sm:px-4 py-2 sm:py-3 whitespace-nowrap">
                                                <div class="text-sm font-medium text-gray-900 truncate max-w-[120px] sm:max-w-none"><?php echo htmlspecialchars($user['full_name']); ?></div>
                                                <div class="text-xs text-gray-500 truncate max-w-[120px] sm:max-w-none"><?php echo htmlspecialchars($user['email']); ?></div>
                                            </td>
                                            <td class="px-3 sm:px-4 py-2 sm:py-3 whitespace-nowrap text-xs sm:text-sm text-gray-900">
                                                <?php echo htmlspecialchars($user['account_number']); ?>
                                            </td>
                                            <td class="px-3 sm:px-4 py-2 sm:py-3 whitespace-nowrap">
                                                <div class="text-sm font-bold text-green-600">
                                                    <?php echo $user['currency']; ?> <?php echo number_format($user['balance'], 2); ?>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="3" class="px-3 sm:px-4 py-6 sm:py-8 text-center text-gray-500">
                                            No user data available
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Period Summary -->
                <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 animate-slide-up card-padding">
                    <h3 class="text-base sm:text-lg font-bold text-gray-900 mb-4">
                        Period Summary: <?php echo date('M j', strtotime($start_date)); ?> - <?php echo date('M j, Y', strtotime($end_date)); ?>
                    </h3>
                    <div class="space-y-3 sm:space-y-4">
                        <div class="flex justify-between items-center p-3 sm:p-4 bg-green-50 rounded-lg">
                            <div>
                                <p class="text-xs sm:text-sm font-medium text-green-800">Total Credits</p>
                                <p class="text-base sm:text-lg font-bold text-green-900">$<?php echo number_format($period_stats['total_credits'] ?? 0, 2); ?></p>
                            </div>
                            <i class="fas fa-arrow-up text-green-600 text-lg sm:text-xl"></i>
                        </div>
                        
                        <div class="flex justify-between items-center p-3 sm:p-4 bg-red-50 rounded-lg">
                            <div>
                                <p class="text-xs sm:text-sm font-medium text-red-800">Total Debits</p>
                                <p class="text-base sm:text-lg font-bold text-red-900">$<?php echo number_format($period_stats['total_debits'] ?? 0, 2); ?></p>
                            </div>
                            <i class="fas fa-arrow-down text-red-600 text-lg sm:text-xl"></i>
                        </div>
                        
                        <div class="flex justify-between items-center p-3 sm:p-4 bg-blue-50 rounded-lg">
                            <div>
                                <p class="text-xs sm:text-sm font-medium text-blue-800">Transactions Count</p>
                                <p class="text-base sm:text-lg font-bold text-blue-900"><?php echo number_format($period_stats['transactions_count'] ?? 0); ?></p>
                            </div>
                            <i class="fas fa-exchange-alt text-blue-600 text-lg sm:text-xl"></i>
                        </div>
                        
                        <div class="flex justify-between items-center p-3 sm:p-4 bg-purple-50 rounded-lg">
                            <div>
                                <p class="text-xs sm:text-sm font-medium text-purple-800">Net Cash Flow</p>
                                <p class="text-base sm:text-lg font-bold <?php echo ($period_stats['net_flow'] ?? 0) >= 0 ? 'text-green-600' : 'text-red-600'; ?>">
                                    $<?php echo number_format($period_stats['net_flow'] ?? 0, 2); ?>
                                </p>
                            </div>
                            <i class="fas fa-chart-line text-purple-600 text-lg sm:text-xl"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Export Options -->
            <div class="bg-white rounded-xl shadow-sm p-4 sm:p-6 mt-6 sm:mt-8 animate-slide-up card-padding">
                <h3 class="text-base sm:text-lg font-bold text-gray-900 mb-4">Export Reports</h3>
                <div class="flex flex-wrap gap-3 sm:gap-4">
                    <button onclick="exportToPDF()" class="bg-red-600 text-white px-4 sm:px-6 py-2 rounded-lg hover:bg-red-700 transition-colors font-medium text-sm sm:text-base">
                        <i class="fas fa-file-pdf mr-2"></i>Export as PDF
                    </button>
                    <button onclick="exportToExcel()" class="bg-green-600 text-white px-4 sm:px-6 py-2 rounded-lg hover:bg-green-700 transition-colors font-medium text-sm sm:text-base">
                        <i class="fas fa-file-excel mr-2"></i>Export as Excel
                    </button>
                    <button onclick="printReport()" class="bg-blue-600 text-white px-4 sm:px-6 py-2 rounded-lg hover:bg-blue-700 transition-colors font-medium text-sm sm:text-base">
                        <i class="fas fa-print mr-2"></i>Print Report
                    </button>
                </div>
            </div>
        </div>
    </main>

    <script>
        // Mobile menu functionality
        document.addEventListener('DOMContentLoaded', function() {
            const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
            const sidebar = document.querySelector('.sidebar');
            const overlay = document.querySelector('.overlay');
            
            function toggleSidebar() {
                sidebar.classList.toggle('active');
                overlay.classList.toggle('active');
                document.body.classList.toggle('overflow-hidden');
            }
            
            if (mobileMenuBtn) {
                mobileMenuBtn.addEventListener('click', toggleSidebar);
            }
            if (overlay) {
                overlay.addEventListener('click', toggleSidebar);
            }
            
            // Handle window resize
            function handleResize() {
                if (window.innerWidth >= 1024) {
                    sidebar.classList.remove('active');
                    overlay.classList.remove('active');
                    document.body.classList.remove('overflow-hidden');
                }
            }
            
            window.addEventListener('resize', handleResize);
        });

        // Transaction Trends Chart
        const transactionCtx = document.getElementById('transactionTrendsChart').getContext('2d');
        const transactionChart = new Chart(transactionCtx, {
            type: 'line',
            data: {
                labels: <?php echo json_encode(array_column($daily_trends, 'date')); ?>,
                datasets: [
                    {
                        label: 'Credits',
                        data: <?php echo json_encode(array_column($daily_trends, 'credits')); ?>,
                        borderColor: '#10b981',
                        backgroundColor: 'rgba(16, 185, 129, 0.1)',
                        tension: 0.4,
                        fill: true
                    },
                    {
                        label: 'Debits',
                        data: <?php echo json_encode(array_column($daily_trends, 'debits')); ?>,
                        borderColor: '#ef4444',
                        backgroundColor: 'rgba(239, 68, 68, 0.1)',
                        tension: 0.4,
                        fill: true
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'top',
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            callback: function(value) {
                                return '$' + value.toLocaleString();
                            }
                        }
                    }
                }
            }
        });

        // Transaction Types Chart
        const typesCtx = document.getElementById('transactionTypesChart').getContext('2d');
        const typesChart = new Chart(typesCtx, {
            type: 'doughnut',
            data: {
                labels: <?php echo json_encode(array_column($transaction_types, 'type')); ?>,
                datasets: [{
                    data: <?php echo json_encode(array_column($transaction_types, 'count')); ?>,
                    backgroundColor: [
                        '#10b981',
                        '#ef4444',
                        '#3b82f6',
                        '#f59e0b'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        // Registration Chart
        const registrationCtx = document.getElementById('registrationChart').getContext('2d');
        const registrationChart = new Chart(registrationCtx, {
            type: 'bar',
            data: {
                labels: <?php echo json_encode(array_column($registration_trends, 'date')); ?>,
                datasets: [{
                    label: 'New Registrations',
                    data: <?php echo json_encode(array_column($registration_trends, 'registrations')); ?>,
                    backgroundColor: '#3b82f6'
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });

        // Currency Distribution Chart
        const currencyCtx = document.getElementById('currencyChart').getContext('2d');
        const currencyChart = new Chart(currencyCtx, {
            type: 'pie',
            data: {
                labels: <?php echo json_encode(array_column($currency_distribution, 'currency')); ?>,
                datasets: [{
                    data: <?php echo json_encode(array_column($currency_distribution, 'account_count')); ?>,
                    backgroundColor: [
                        '#10b981',
                        '#3b82f6',
                        '#f59e0b',
                        '#ef4444'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        // Support Tickets Chart
        const ticketsCtx = document.getElementById('ticketsChart').getContext('2d');
        const ticketsChart = new Chart(ticketsCtx, {
            type: 'polarArea',
            data: {
                labels: <?php echo json_encode(array_column($ticket_stats, 'status')); ?>,
                datasets: [{
                    data: <?php echo json_encode(array_column($ticket_stats, 'count')); ?>,
                    backgroundColor: [
                        '#ef4444',
                        '#f59e0b',
                        '#10b981',
                        '#6b7280'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });

        // Export functions
        function exportToPDF() {
            alert('PDF export functionality would be implemented here');
            // In a real implementation, this would generate a PDF report
        }

        function exportToExcel() {
            alert('Excel export functionality would be implemented here');
            // In a real implementation, this would generate an Excel report
        }

        function printReport() {
            window.print();
        }
    </script>

</body>
</html>